const generateStudentTimeSlots = (student, calendar) => {
    const slots = [];
    const studentStart = dayjs(student.startDate).startOf('day');
    let startIdx = calendar.findIndex(d => d.date.isSame(studentStart, 'day') || d.date.isAfter(studentStart));
    if (startIdx === -1) return [];

    let slotCounter = 0;
    for (let i = startIdx; i < calendar.length; i++) {
        const day = calendar[i];
        let activeConfig;
        if (student.scheduleHistory && student.scheduleHistory.length > 0) {
            const targetDate = day.date;
            activeConfig = student.scheduleHistory
                .filter(c => targetDate.isSame(dayjs(c.startDate), 'day') || targetDate.isAfter(dayjs(c.startDate)))
                .sort((a, b) => dayjs(b.startDate).valueOf() - dayjs(a.startDate).valueOf())[0];
        }
        if (!activeConfig) activeConfig = { id: 'def', startDate: student.startDate, trainingType: 'FULL_TIME' };

        const blocksCount = day.hours || 6;
        let daySlots = Array(blocksCount).fill(true);

        if (activeConfig.trainingType === 'PART_TIME' && activeConfig.schedule) {
            const dayOfWeek = day.date.day();
            const sched = activeConfig.schedule[dayOfWeek];
            if (sched && sched.slots) daySlots = sched.slots;
            else daySlots = Array(blocksCount).fill(false);
        }

        daySlots.forEach((isActive, sIdx) => {
            if (isActive && sIdx < blocksCount) {
                slots.push({ id: slotCounter++, dateStr: day.date.format('YYYY-MM-DD'), date: day.date, slotIndex: sIdx });
            }
        });
    }
    return slots;
};

window.getEffectiveDailyHours = (student, day) => {
    let activeConfig;
    if (student.scheduleHistory && student.scheduleHistory.length > 0) {
        const targetDate = day.date;
        activeConfig = student.scheduleHistory
            .filter(c => targetDate.isSame(dayjs(c.startDate), 'day') || targetDate.isAfter(dayjs(c.startDate)))
            .sort((a, b) => dayjs(b.startDate).valueOf() - dayjs(a.startDate).valueOf())[0];
    }
    if (!activeConfig) activeConfig = { id: 'legacy', startDate: student.startDate, trainingType: 'FULL_TIME' };
    const blocksCount = day.hours || 6;
    if (activeConfig.trainingType === 'PART_TIME' && activeConfig.schedule) {
        const dayOfWeek = day.date.day();
        const sched = activeConfig.schedule[dayOfWeek];
        if (sched && sched.slots) return sched.slots.filter((s, idx) => s && idx < blocksCount).length;
        return 0;
    }
    return blocksCount; 
};

window.calculateStudentProgress = (student, calendar) => {
  const startDate = dayjs(student.startDate).startOf('day');
  const today = dayjs().startOf('day');
  const firstWorkDayIndex = calendar.findIndex(d => d.date.isSame(startDate, 'day') || d.date.isAfter(startDate));
  if (firstWorkDayIndex === -1) return { actualStartDate: null, hoursWorked: 0, daysWorked: 0, firstWorkDayIndex: -1 };
  const actualStartDate = calendar[firstWorkDayIndex].date;
  let hoursWorked = 0, daysWorked = 0;
  for (let i = firstWorkDayIndex; i < calendar.length; i++) {
    const day = calendar[i];
    if (day.date.isAfter(today)) break;
    const effectiveHours = window.getEffectiveDailyHours(student, day); 
    if (effectiveHours > 0) { hoursWorked += effectiveHours; daysWorked++; }
  }
  return { actualStartDate, hoursWorked, daysWorked, firstWorkDayIndex };
};

window.calculateDynamicPlanning = (student, calendar, matrice, options = {}) => {
    const availableSlots = generateStudentTimeSlots(student, calendar);
    const totalSlotsCount = availableSlots.length;
    const occupiedSlots = new Set();
    const allocationsByDate = {};
    const skillStatsMap = new Map();

    if (student.examEvents) {
        student.examEvents.forEach(exam => {
            let hRem = exam.duration;
            for (let i = 0; i < availableSlots.length; i++) {
                const s = availableSlots[i];
                if (s.dateStr === exam.date && hRem > 0 && !occupiedSlots.has(s.id)) {
                    occupiedSlots.add(s.id); hRem--;
                    if (!allocationsByDate[s.dateStr]) allocationsByDate[s.dateStr] = [];
                    let ex = allocationsByDate[s.dateStr].find(a => a.code === exam.competencyCode && a.isExam);
                    if (ex) ex.hours++; else allocationsByDate[s.dateStr].push({ code: exam.competencyCode, hours: 1, isExam: true });
                }
            }
        });
    }

    if (student.internshipEvents) {
        student.internshipEvents.forEach(intern => {
            availableSlots.forEach(s => { if (s.dateStr === intern.date) occupiedSlots.add(s.id); });
            if (!allocationsByDate[intern.date]) allocationsByDate[intern.date] = [];
            allocationsByDate[intern.date].push({ code: intern.competencyCode, hours: intern.duration, isInternship: true });
        });
    }

    const order = matrice.slice().sort((a, b) => {
        const da = student.evaluations?.[a.code]?.realStartDate ? dayjs(student.evaluations[a.code].realStartDate).valueOf() : Infinity;
        const db = student.evaluations?.[b.code]?.realStartDate ? dayjs(student.evaluations[b.code].realStartDate).valueOf() : Infinity;
        return da - db;
    });

    order.forEach(comp => {
        const evalData = student.evaluations?.[comp.code];
        const hExams = (student.examEvents || []).filter(e => e.competencyCode === comp.code).reduce((sum, e) => sum + e.duration, 0);
        const hInterns = (student.internshipEvents || []).filter(e => e.competencyCode === comp.code).reduce((sum, e) => sum + e.duration, 0);
        
        let hNeeded = Math.max(0, comp.hours - hExams - hInterns);
        
        let hAlloc = 0, firstSlot = null, lastSlot = null, searchIndex = 0;
        // MODIFICATION : matrix par défaut
        const paceMode = options.forceMatrixPace ? 'matrix' : (evalData?.paceMode || 'matrix');
        const matrixSequence = comp.weeklyHours.filter(h => h > 0);
        const lastKnownQuota = matrixSequence.length > 0 ? matrixSequence[matrixSequence.length - 1] : 30;

        let currentCompWorkDayIdx = -1, lastDayStr = "", hoursToday = 0, hoursThisMatrixWeek = 0;

        if (evalData?.realStartDate) {
            const forced = dayjs(evalData.realStartDate).startOf('day');
            searchIndex = availableSlots.findIndex(s => s.date.isSame(forced, 'day') || s.date.isAfter(forced, 'day'));
            if (searchIndex === -1) searchIndex = totalSlotsCount;
        }

        const limitDate = evalData?.realEndDate ? dayjs(evalData.realEndDate).endOf('day') : null;

        while (searchIndex < totalSlotsCount && hAlloc < hNeeded) {
            const cur = availableSlots[searchIndex];
            
            if (cur.dateStr !== lastDayStr) {
                currentCompWorkDayIdx++;
                lastDayStr = cur.dateStr; hoursToday = 0;
                if (currentCompWorkDayIdx > 0 && currentCompWorkDayIdx % 5 === 0) hoursThisMatrixWeek = 0;
            }

            if (occupiedSlots.has(cur.id)) { searchIndex++; continue; }
            if (limitDate && cur.date.isAfter(limitDate)) break; 

            if (paceMode === 'matrix') {
                const weekIdx = Math.floor(currentCompWorkDayIdx / 5);
                const weeklyQuota = matrixSequence[weekIdx] ?? lastKnownQuota;
                const dailyLimit = Math.ceil(weeklyQuota / 5);
                
                if (hoursThisMatrixWeek >= weeklyQuota || hoursToday >= dailyLimit) { 
                    while (searchIndex < totalSlotsCount && availableSlots[searchIndex].dateStr === lastDayStr) {
                        searchIndex++;
                    }
                    continue; 
                }
            }

            if (hAlloc === 0) firstSlot = cur;
            lastSlot = cur; occupiedSlots.add(cur.id); 
            hAlloc++; hoursToday++; hoursThisMatrixWeek++;
            if (!allocationsByDate[cur.dateStr]) allocationsByDate[cur.dateStr] = [];
            let a = allocationsByDate[cur.dateStr].find(x => x.code === comp.code && !x.isExam && !x.isInternship);
            if (a) a.hours++; else allocationsByDate[cur.dateStr].push({ code: comp.code, hours: 1 });
            searchIndex++;
        }

        let absoluteLastDate = lastSlot?.date || null;
        Object.entries(allocationsByDate).forEach(([dateStr, allocs]) => {
            if (allocs.some(a => a.code === comp.code)) {
                const d = dayjs(dateStr);
                if (!absoluteLastDate || d.isAfter(absoluteLastDate)) {
                    absoluteLastDate = d;
                }
            }
        });

        skillStatsMap.set(comp.code, {
            ...comp,
            dynamicStart: evalData?.realStartDate ? dayjs(evalData.realStartDate) : (firstSlot?.date || null),
            dynamicEnd: (evalData?.status === 'completed' && evalData?.realEndDate) ? dayjs(evalData.realEndDate) : absoluteLastDate,
            status: evalData?.status || 'upcoming',
            evaluation: evalData
        });
    });

    return { skillStats: matrice.map(c => skillStatsMap.get(c.code)), allocationsByDate };
};

window.calculateEndDate = (student, calendar, matrice, targetCompCode, startDateStr, paceMode, exams, internships) => {
    const targetComp = matrice.find(c => c.code === targetCompCode);
    if (!targetComp) return '';
    const tempStudent = {
        ...student,
        evaluations: { 
            ...student.evaluations, 
            [targetCompCode]: { code: targetCompCode, status: 'in_progress', realStartDate: startDateStr, paceMode } 
        },
        examEvents: exams || student.examEvents,
        internshipEvents: internships || student.internshipEvents
    };
    const plan = window.calculateDynamicPlanning(tempStudent, calendar, matrice);
    const stat = plan.skillStats.find(s => s.code === targetCompCode);
    return stat?.dynamicEnd ? stat.dynamicEnd.format('YYYY-MM-DD') : '';
};
window.calculateProjectedPlanning = (student, calendar, matrice) => {
    const res = window.calculateDynamicPlanning({ ...student, evaluations: {}, examEvents: [], internshipEvents: [] }, calendar, matrice, { forceMatrixPace: true });
    return { ...res, skillStats: res.skillStats.map(s => ({ ...s, projectedStart: s.dynamicStart, projectedEnd: s.dynamicEnd })) };
};
