
// Initialisation des plugins DayJS (Global)
dayjs.extend(window.dayjs_plugin_customParseFormat);
dayjs.extend(window.dayjs_plugin_utc);
dayjs.extend(window.dayjs_plugin_isoWeek);
dayjs.extend(window.dayjs_plugin_isSameOrBefore);
dayjs.extend(window.dayjs_plugin_isSameOrAfter);
dayjs.locale('fr');

const FRENCH_MONTHS = {
  'janvier': '01', 'janv.': '01', 'janv': '01',
  'février': '02', 'fevrier': '02', 'févr.': '02', 'fevr.': '02', 'févr': '02', 'fevr': '02',
  'mars': '03', 'mar': '03', 'mar.': '03',
  'avril': '04', 'avr.': '04', 'avr': '04',
  'mai': '05',
  'juin': '06',
  'juillet': '07', 'juil.': '07', 'juil': '07',
  'août': '08', 'aout': '08',
  'septembre': '09', 'sept.': '09', 'sept': '09',
  'octobre': '10', 'oct.': '10', 'oct': '10',
  'novembre': '11', 'nov.': '11', 'nov': '11',
  'décembre': '12', 'decembre': '12', 'déc.': '12', 'dec.': '12', 'déc': '12', 'dec': '12'
};

const DEFAULT_SCHEDULE = {
  1: { am: true, pm: true },
  2: { am: true, pm: true },
  3: { am: true, pm: true },
  4: { am: true, pm: true },
  5: { am: true, pm: true }
};

window.parseCsv = (csv) => {
  const lines = csv.split(/\r\n|\n|\r/).filter(line => line.trim() !== '');
  if (lines.length === 0) return [];
  const header = lines[0];
  const semicolonCount = (header.match(/;/g) || []).length;
  const commaCount = (header.match(/,/g) || []).length;
  const delimiter = semicolonCount > commaCount ? ';' : ',';
  return lines.map(line => {
    return line.split(delimiter).map(cell => {
      let val = cell.trim();
      if (val.startsWith('"') && val.endsWith('"')) {
        val = val.substring(1, val.length - 1);
      }
      return val.trim();
    });
  });
};

window.parseCalendarCsv = (csv, defaultHours) => {
  const data = window.parseCsv(csv);
  return data.slice(1).map(row => {
    if (row.length < 3) return null;
    const day = row[0].trim();
    const month = row[1].trim();
    const year = row[2].trim();
    const normalizedMonth = month.toLowerCase();
    const monthNum = FRENCH_MONTHS[normalizedMonth];
    let date;
    if (monthNum) {
       date = dayjs(`${year}-${monthNum}-${day.padStart(2, '0')}`);
    } else {
       date = dayjs(`${day} ${month} ${year}`, 'D MMMM YYYY', 'fr');
       if (!date.isValid()) {
          date = dayjs(`${year}-${month}-${day}`);
       }
    }
    if (!date.isValid()) return null;
    return {
      date: date,
      hours: 6
    };
  }).filter(d => d !== null && d.date.isValid())
    .sort((a, b) => a.date.valueOf() - b.date.valueOf());
};

window.parseStudentsCsv = (csv) => {
  const data = window.parseCsv(csv);
  const rows = data.slice(1);
  return rows.map(row => {
    if (row.length < 2) return null;
    const fullName = row[0];
    const dateStr = row[1];
    const parts = fullName.split(' ');
    const firstName = parts[0];
    const lastName = parts.slice(1).join(' ');
    const date = dayjs(dateStr);
    if (!date.isValid()) return null;
    const initialHistory = {
        id: crypto.randomUUID(),
        startDate: dateStr,
        trainingType: 'FULL_TIME',
        schedule: DEFAULT_SCHEDULE
    };
    return {
      id: crypto.randomUUID(),
      firstName,
      lastName,
      name: fullName,
      startDate: dateStr,
      trainingType: 'FULL_TIME',
      schedule: DEFAULT_SCHEDULE,
      scheduleHistory: [initialHistory]
    };
  }).filter(s => s !== null);
};

window.processMatrice = (data) => {
  const fixedCols = 6;
  const rows = data.slice(1);
  return rows.map(row => {
    if (row.length < fixedCols) return null;
    const weeklyHours = row.slice(fixedCols).map(h => parseInt(h, 10) || 0);
    
    const startWeek = weeklyHours.findIndex(h => h > 0);
    let endWeek = -1;
    for (let i = weeklyHours.length - 1; i >= 0; i--) {
      if (weeklyHours[i] > 0) {
        endWeek = i;
        break;
      }
    }

    const number = parseInt(row[0], 10);
    let hours = parseInt(row[3], 10);
    if (isNaN(number) || isNaN(hours)) return null;
    
    const dailyHours = [];
    weeklyHours.forEach(h => {
        if (h > 0) {
            for (let i = 0; i < h; i++) {
                dailyHours.push(1);
            }
        }
    });

    return {
      number,
      code: row[1],
      name: row[2],
      hours,
      type: row[4],
      local: row[5],
      weeklyHours,
      dailyHours,
      startWeek,
      endWeek
    };
  }).filter(c => c !== null && c.hours > 0 && c.startWeek !== -1);
};

window.generateProjectFile = (students, calendar, matrix, defaultDailyHours, projectName, isDynamic, settings) => {
  const data = {
    version: "1.5",
    projectName,
    students,
    calendar: Array.isArray(calendar) ? calendar.map(c => ({
        date: c.date.format('YYYY-MM-DD'),
        hours: c.hours || 6
    })) : [],
    matrix,
    settings: settings || {
      defaultDailyHours: 6,
      isDynamic
    }
  };
  return JSON.stringify(data, null, 2);
};

window.parseProjectFile = (jsonString) => {
  const data = JSON.parse(jsonString);
  let loadedCalendar = [];
  if (Array.isArray(data.calendar)) {
      loadedCalendar = data.calendar.map(c => ({
          date: dayjs(c.date),
          hours: c.hours || 6
      }));
  } else if (data.calendar && typeof data.calendar === 'object' && Array.isArray(data.calendar.dates)) {
      loadedCalendar = data.calendar.dates.map(dateStr => ({
          date: dayjs(dateStr),
          hours: 6
      }));
  }
  
  let globalTosEmail = data.settings?.tosEmail || '';
  if (!globalTosEmail && data.students.length > 0) {
      globalTosEmail = data.students.find(s => s.tosEmail)?.tosEmail || '';
  }

  return {
    projectName: data.projectName || "Macroplanification FP",
    students: data.students.map(s => {
       let history = s.scheduleHistory;
       if (!history || history.length === 0) {
          history = [{
              id: crypto.randomUUID(),
              startDate: s.startDate,
              trainingType: s.trainingType || 'FULL_TIME',
              schedule: s.schedule || DEFAULT_SCHEDULE
          }];
       }
       return {
         ...s,
         name: `${s.firstName} ${s.lastName}`,
         scheduleHistory: history,
         studentEmail: s.studentEmail || ''
       };
    }),
    calendar: loadedCalendar.sort((a, b) => a.date.valueOf() - b.date.valueOf()),
    matrix: data.matrix,
    settings: {
        ...data.settings,
        defaultDailyHours: 6,
        isDynamic: data.settings?.isDynamic ?? false,
        tosEmail: globalTosEmail
    }
  };
};
