const ProgressionDetailModal = ({ isOpen, onClose, onSave, competency, initialEval, initialExams, initialInternships, student, fullMatrice, calendar }) => {
  const { useState, useEffect, useMemo } = React;
  
  const [status, setStatus] = useState('in_progress');
  const [paceMode, setPaceMode] = useState('matrix');
  const [result, setResult] = useState(undefined);
  const [realStartDate, setRealStartDate] = useState('');
  const [realEndDate, setRealEndDate] = useState('');
  const [resultDate, setResultDate] = useState('');
  const [comment, setComment] = useState('');
  const [exams, setExams] = useState([]);
  const [internships, setInternships] = useState([]);
  const [newEvDate, setNewEvDate] = useState(dayjs().format('YYYY-MM-DD'));
  const [newEvHours, setNewEvHours] = useState(3);
  const [newEvComment, setNewEvComment] = useState('');

  useEffect(() => {
    if (isOpen && competency) {
       setStatus(initialEval?.status || 'in_progress');
       setPaceMode(initialEval?.paceMode || 'matrix');
       setResult(initialEval?.result);
       setRealStartDate(initialEval?.realStartDate || competency.dynamicStart?.format('YYYY-MM-DD') || '');
       setRealEndDate(initialEval?.realEndDate || competency.dynamicEnd?.format('YYYY-MM-DD') || '');
       setResultDate(initialEval?.resultDate || dayjs().format('YYYY-MM-DD'));
       setComment(initialEval?.comment || '');
       setExams(initialExams || []);
       setInternships(initialInternships || []);
    }
  }, [isOpen, competency, initialEval, initialExams, initialInternships]);

  useEffect(() => {
      if (isOpen && competency && student && status !== 'completed') {
          const newEnd = window.calculateEndDate(
              student, 
              calendar, 
              fullMatrice, 
              competency.code, 
              realStartDate, 
              paceMode, 
              exams, 
              internships
          );
          if (newEnd && newEnd !== realEndDate) {
              setRealEndDate(newEnd);
          }
      }
  }, [paceMode, realStartDate, exams, internships, isOpen, status, competency]);

  const lastActivityDate = useMemo(() => {
      let latest = dayjs(realEndDate);
      [...exams, ...internships].forEach(ev => {
          const d = dayjs(ev.date);
          if (d.isAfter(latest)) latest = d;
      });
      return latest;
  }, [realEndDate, exams, internships]);

  if (!isOpen || !competency) return null;

  const dayCapacity = calendar.find(c => c.date.format('YYYY-MM-DD') === newEvDate)?.hours || 6;

  return (
    <div className="modal show d-block" style={{ backgroundColor: 'rgba(0,0,0,0.7)', zIndex: 100000 }}>
      <div className="modal-dialog modal-dialog-centered modal-lg shadow-lg">
        <div className="modal-content border-0 overflow-hidden" style={{ maxHeight: '95vh' }}>
          <div className="modal-header bg-dark text-white border-0 py-3">
            <div>
                <h5 className="modal-title fw-bold mb-0">Gestion : {competency.name}</h5>
                <div className="badge bg-light text-dark mt-1">C-{competency.number} • {competency.hours}h</div>
            </div>
            <button type="button" className="btn-close btn-close-white" onClick={onClose}></button>
          </div>
          
          <div className="modal-body p-0 overflow-hidden">
              <div className="row g-0 h-100" style={{ minHeight: '520px' }}>
                  <div className="col-md-6 p-4 border-end bg-white overflow-y-auto" style={{ maxHeight: '75vh' }}>
                    <h6 className="fw-black text-secondary text-uppercase mb-3" style={{fontSize: '0.7rem'}}>Apprentissage (Cours)</h6>
                    
                    <div className="mb-4">
                        <label className="form-label small fw-bold text-muted">Statut</label>
                        <div className="d-flex gap-2">
                            <input type="radio" className="btn-check" id="st_in" name="status" checked={status === 'in_progress'} onChange={() => setStatus('in_progress')} />
                            <label className="btn btn-outline-primary w-50 fw-bold border-2" htmlFor="st_in">EN COURS</label>
                            <input type="radio" className="btn-check" id="st_cp" name="status" checked={status === 'completed'} onChange={() => setStatus('completed')} />
                            <label className="btn btn-outline-success w-50 fw-bold border-2" htmlFor="st_cp">TERMINÉ</label>
                        </div>
                    </div>

                    <div className="mb-4">
                        <label className="form-label small fw-bold text-muted">Rythme de calcul</label>
                        <select className="form-select form-select-sm fw-bold border-primary border-opacity-25" value={paceMode} onChange={e => setPaceMode(e.target.value)}>
                            <option value="matrix">Selon la matrice officielle (Recommandé)</option>
                            <option value="fast">Combler l'horaire (Tetris)</option>
                        </select>
                    </div>

                    <div className="row g-3 mb-4">
                        <div className="col-6">
                            <label className="small fw-bold text-muted">Début réel</label>
                            <input type="date" value={realStartDate} onChange={e => setRealStartDate(e.target.value)} className="form-control form-control-sm" />
                        </div>
                        <div className="col-6">
                            <label className="small fw-bold text-muted">Fin des cours</label>
                            <input type="date" value={realEndDate} readOnly className="form-control form-control-sm bg-light fw-bold text-primary" />
                        </div>
                    </div>

                    <div className="p-3 bg-info bg-opacity-10 rounded border border-info border-opacity-25 mb-4 shadow-sm">
                        <div className="d-flex justify-content-between align-items-center mb-1">
                            <span className="small fw-bold text-info text-uppercase" style={{fontSize:'0.65rem'}}>Fin compétence (Dernière activité)</span>
                            <i className="bi bi-calendar-event text-info"></i>
                        </div>
                        <div className="h5 fw-black text-dark mb-0">{lastActivityDate.format('D MMMM YYYY')}</div>
                    </div>

                    {status === 'completed' && (
                        <div className="mb-4 p-3 bg-light rounded border border-success border-opacity-25 shadow-sm text-center">
                            <label className="form-label small fw-bold text-success text-uppercase mb-2 d-block" style={{fontSize:'0.65rem'}}>Validation Finale</label>
                            <div className="d-flex gap-2 mb-3">
                                <input type="radio" className="btn-check" id="res_ok" name="result" checked={result === 'success'} onChange={() => setResult('success')} />
                                <label className="btn btn-outline-success flex-grow-1 fw-bold" htmlFor="res_ok">RÉUSSITE</label>
                                <input type="radio" className="btn-check" id="res_ko" name="result" checked={result === 'failure'} onChange={() => setResult('failure')} />
                                <label className="btn btn-outline-danger flex-grow-1 fw-bold" htmlFor="res_ko">ÉCHEC</label>
                            </div>
                            <input type="date" value={resultDate} onChange={e => setResultDate(e.target.value)} className="form-control form-control-sm shadow-sm" />
                        </div>
                    )}

                    <div className="mb-0"><label className="small fw-bold text-muted">Observations</label><textarea value={comment} onChange={e => setComment(e.target.value)} className="form-control" rows={2} placeholder="Observations..." /></div>
                  </div>

                  <div className="col-md-6 bg-light d-flex flex-column h-100 overflow-hidden">
                    <div className="p-4 flex-grow-1 overflow-auto mb-2 pe-3">
                        <h6 className="fw-black text-primary text-uppercase mb-3" style={{fontSize: '0.7rem'}}><i className="bi bi-mortarboard-fill"></i> Épreuves ({exams.length})</h6>
                        <div className="vstack gap-2 mb-4">
                            {exams.map(ex => (
                                <div key={ex.id} className="p-2 rounded bg-white border shadow-sm d-flex flex-column gap-1">
                                    <div className="d-flex justify-content-between align-items-center">
                                        <div className="small fw-bold text-dark">{dayjs(ex.date).format('D MMM YYYY')} <span className="text-muted fw-normal">• {ex.duration}h</span></div>
                                        <button type="button" onClick={() => setExams(exams.filter(e => e.id !== ex.id))} className="btn btn-sm text-danger p-0 border-0 shadow-none"><i className="bi bi-trash3-fill"></i></button>
                                    </div>
                                    {ex.comment && <div className="small text-muted fst-italic border-top pt-1 mt-1">{ex.comment}</div>}
                                </div>
                            ))}
                        </div>

                        <h6 className="fw-black text-purple text-uppercase mb-3" style={{fontSize: '0.7rem'}}><i className="bi bi-building"></i> Stages ({internships.length})</h6>
                        <div className="vstack gap-2">
                            {internships.map(inT => (
                                <div key={inT.id} className="p-2 rounded bg-white border shadow-sm d-flex flex-column gap-1">
                                    <div className="d-flex justify-content-between align-items-center">
                                        <div className="small fw-bold text-dark">{dayjs(inT.date).format('D MMM YYYY')} <span className="text-muted fw-normal">• {inT.duration}h</span></div>
                                        <button type="button" onClick={() => setInternships(internships.filter(i => i.id !== inT.id))} className="btn btn-sm text-danger p-0 border-0 shadow-none"><i className="bi bi-trash3-fill"></i></button>
                                    </div>
                                    {inT.comment && <div className="small text-muted fst-italic border-top pt-1 mt-1">{inT.comment}</div>}
                                </div>
                            ))}
                        </div>
                    </div>

                    <div className="p-3 bg-white border-top shadow-sm mt-auto">
                        <h6 className="small fw-black text-dark text-uppercase mb-2" style={{ fontSize: '0.65rem' }}>Ajouter un événement fixe</h6>
                        <div className="vstack gap-2">
                            <div className="row g-2">
                                <div className="col-7"><input type="date" value={newEvDate} onChange={e => setNewEvDate(e.target.value)} className="form-control form-control-sm shadow-none" /></div>
                                <div className="col-5">
                                    <select className="form-select form-select-sm shadow-none" value={newEvHours} onChange={e => setNewEvHours(parseInt(e.target.value))}>
                                        {Array.from({length: 10}).map((_,i) => <option key={i+1} value={i+1}>{i+1}h</option>)}
                                    </select>
                                </div>
                            </div>
                            <input type="text" className="form-control form-control-sm" placeholder="Commentaire (Lieu, notes...)" value={newEvComment} onChange={e => setNewEvComment(e.target.value)} />
                            <div className="d-flex gap-2">
                                <button type="button" onClick={() => { 
                                    const h = Math.min(newEvHours, dayCapacity);
                                    setExams([...exams, { id: crypto.randomUUID(), competencyCode: competency.code, date: newEvDate, duration: h, comment: newEvComment }]); 
                                    setNewEvComment(''); 
                                }} className="btn btn-primary btn-sm fw-bold flex-grow-1 shadow-sm"><i className="bi bi-plus-lg"></i> Épreuve</button>
                                <button type="button" onClick={() => { 
                                    setInternships([...internships, { id: crypto.randomUUID(), competencyCode: competency.code, date: newEvDate, duration: newEvHours, comment: newEvComment }]); 
                                    setNewEvComment(''); 
                                }} className="btn btn-purple text-white btn-sm fw-bold flex-grow-1 shadow-sm" style={{backgroundColor: '#6f42c1'}}><i className="bi bi-plus-lg"></i> Stage</button>
                            </div>
                            {newEvHours > dayCapacity && <div className="small text-danger fst-italic" style={{fontSize:'0.6rem'}}>* Épreuve plafonnée à {dayCapacity}h pour cette date.</div>}
                        </div>
                    </div>
                  </div>
              </div>
          </div>

          <div className="modal-footer bg-light border-0 d-flex justify-content-between p-3 flex-shrink-0">
            <button type="button" onClick={() => { if (confirm("Réinitialiser tout le suivi de ce module ?")) { onSave({ evaluation: null, exams: [], internships: [] }); onClose(); } }} className="btn btn-outline-danger btn-sm fw-bold px-3 border-2 shadow-none"><i className="bi bi-arrow-counterclockwise"></i> Réinitialiser</button>
            <div className="d-flex gap-2">
                <button type="button" onClick={onClose} className="btn btn-light fw-bold px-4 border shadow-none">Annuler</button>
                <button type="button" 
                    onClick={() => {
                        onSave({ 
                            evaluation: { code: competency.code, status, paceMode, result: status === 'completed' ? result : undefined, resultDate, realStartDate, realEndDate, comment }, 
                            exams, 
                            internships 
                        });
                        onClose();
                    }} 
                    className="btn btn-dark fw-bold px-5 shadow-sm text-white border-0"
                >Valider</button>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

const SkillProgression = ({ student, calendar, matrice, onUpdateStudent, isDynamic }) => {
  const { useState, useEffect, useMemo } = React;
  const { calculateDynamicPlanning } = window;
  const [modalOpen, setModalOpen] = useState(false);
  const [selectedComp, setSelectedComp] = useState(null);

  const fullMatrice = useMemo(() => student ? [...matrice, ...(student.customCompetencies || [])] : matrice, [matrice, student]);
  const planning = useMemo(() => student ? calculateDynamicPlanning(student, calendar, fullMatrice) : null, [student, calendar, fullMatrice]);

  if (!student) return <div className="p-5 text-center text-muted border rounded bg-light m-4 shadow-sm">Sélectionnez un étudiant.</div>;
  
  return (
    <div className="h-100 d-flex flex-column">
      <div className="mb-4 p-3 bg-white border rounded shadow-sm d-flex justify-content-between align-items-center">
        <h2 className="h5 fw-bold mb-0 text-dark">{student.name}</h2>
      </div>
      <div className="table-responsive bg-white rounded shadow-sm border flex-grow-1 overflow-auto">
        <table className="table table-hover mb-0 align-middle">
          <thead className="bg-light text-secondary small text-uppercase"><tr><th className="px-4 py-3">Module</th><th className="px-4 py-3 text-center">Dates Apprentissage</th><th className="px-4 py-3 text-center">Rythme</th><th className="px-4 py-3 text-center">Statut</th>{isDynamic && <th className="px-4 py-3 text-center">Actions</th>}</tr></thead>
          <tbody>
            {planning?.skillStats.map(skill => {
                const evaluation = student.evaluations?.[skill.code];
                const pace = evaluation?.paceMode || 'matrix';
                const exList = (student.examEvents || []).filter(e => e.competencyCode === skill.code).sort((a,b)=>dayjs(a.date).valueOf()-dayjs(b.date).valueOf());
                const inList = (student.internshipEvents || []).filter(e => e.competencyCode === skill.code).sort((a,b)=>dayjs(a.date).valueOf()-dayjs(b.date).valueOf());
                const exCount = exList.length;
                const inCount = inList.length;

                const exTooltip = exCount > 0 
                  ? `Épreuve(s) prévue(s) :\n${exList.map(e => `• ${dayjs(e.date).format('D MMM')} : ${e.duration}h ${e.comment ? `(${e.comment})` : ''}`).join('\n')}`
                  : '';
                const inTooltip = inCount > 0 
                  ? `Stage(s) prévu(s) :\n${inList.map(e => `• ${dayjs(e.date).format('D MMM')} : ${e.duration}h ${e.comment ? `(${e.comment})` : ''}`).join('\n')}`
                  : '';

                return (
                    <tr key={skill.code}>
                        <td className="px-4 py-3"><div className="fw-bold text-dark">{skill.name}</div><small className="text-muted">C-{skill.number} • {skill.hours}h</small></td>
                        <td className="px-4 py-3 text-center small fw-bold text-dark">{skill.dynamicStart ? skill.dynamicStart.format('DD MMM YY') : '--'} au {skill.dynamicEnd ? skill.dynamicEnd.format('DD MMM YY') : '--'}</td>
                        <td className="px-4 py-3 text-center">
                            <span className={`badge ${pace === 'matrix' ? 'bg-info bg-opacity-10 text-info border-info' : 'bg-warning bg-opacity-10 text-warning border-warning'} border small text-uppercase`} style={{fontSize:'0.6rem'}}>
                                {pace === 'matrix' ? 'Matrice' : 'Tetris'}
                            </span>
                        </td>
                        <td className="px-4 py-3 text-center">
                            <div className="vstack gap-1 align-items-center">
                                <span className={`badge ${skill.status==='completed'?'bg-success':skill.status==='in_progress'?'bg-primary':'bg-light text-dark border'}`} style={{fontSize:'0.6rem'}}>{skill.status==='in_progress'?'EN COURS':skill.status==='completed'?'TERMINÉ':'À VENIR'}</span>
                                {(exCount > 0 || inCount > 0) && (
                                    <div className="d-flex gap-1 mt-1">
                                        {exCount > 0 && <span className="badge bg-primary text-white border border-primary shadow-sm cursor-help" data-bs-toggle="tooltip" title={exTooltip} style={{fontSize:'0.55rem', padding: '0.35em 0.55em'}}>🎓 {exCount}</span>}
                                        {inCount > 0 && <span className="badge text-white border shadow-sm cursor-help" data-bs-toggle="tooltip" title={inTooltip} style={{fontSize:'0.55rem', padding: '0.35em 0.55em', backgroundColor: '#6f42c1', borderColor: '#6f42c1'}}>🏢 {inCount}</span>}
                                    </div>
                                )}
                            </div>
                        </td>
                        {isDynamic && <td className="px-4 py-3 text-center"><button onClick={()=>{setSelectedComp(skill); setModalOpen(true);}} className="btn btn-sm btn-outline-dark fw-bold border-2 px-3">Gérer</button></td>}
                    </tr>
                );
            })}
          </tbody>
        </table>
      </div>
      <ProgressionDetailModal 
          isOpen={modalOpen} 
          onClose={() => setModalOpen(false)} 
          onSave={(bundle) => {
              const fEx = (student.examEvents || []).filter(e => e.competencyCode !== selectedComp.code);
              const fIn = (student.internshipEvents || []).filter(e => e.competencyCode !== selectedComp.code);
              const newEvals = { ...student.evaluations };
              if (bundle.evaluation) {
                  newEvals[bundle.evaluation.code] = bundle.evaluation;
                  onUpdateStudent({ ...student, evaluations: newEvals, examEvents: [...fEx, ...bundle.exams], internshipEvents: [...fIn, ...bundle.internships] });
              } else {
                  delete newEvals[selectedComp.code];
                  onUpdateStudent({ ...student, evaluations: newEvals, examEvents: fEx, internshipEvents: fIn });
              }
          }} 
          competency={selectedComp} 
          initialEval={student.evaluations?.[selectedComp?.code]} 
          initialExams={(student.examEvents || []).filter(e => e.competencyCode === selectedComp?.code)}
          initialInternships={(student.internshipEvents || []).filter(e => e.competencyCode === selectedComp?.code)}
          student={student} fullMatrice={fullMatrice} calendar={calendar}
      />
    </div>
  );
};
window.SkillProgression = SkillProgression;