const RoomOccupancy = ({ students, calendar, matrice, isDynamic }) => {
  const { useState, useMemo, useEffect } = React;
  const { calculateStudentProgress, calculateDynamicPlanning } = window;

  const [selectedRoom, setSelectedRoom] = useState(matrice[0]?.local || '');
  const [viewDate, setViewDate] = useState(dayjs().startOf('month'));
  const [selectedDayStr, setSelectedDayStr] = useState(null);
  const [statusFilter, setStatusFilter] = useState('all'); // 'all' | 'in_progress' | 'upcoming'

  const rooms = useMemo(() => Array.from(new Set(matrice.map(m => m.local))).sort(), [matrice]);
  
  useEffect(() => {
    if (window.bootstrap) {
      const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
      tooltipTriggerList.map(el => new bootstrap.Tooltip(el));
    }
    return () => {
      const tooltips = document.querySelectorAll('.tooltip');
      tooltips.forEach(t => t.remove());
    };
  }, [viewDate, selectedRoom, students, isDynamic, statusFilter]);

  const monthsList = useMemo(() => {
    if (calendar.length === 0) return [];
    const start = calendar[0].date.startOf('month');
    const end = calendar[calendar.length - 1].date.startOf('month');
    const list = [];
    let curr = start;
    while(curr.isBefore(end) || curr.isSame(end, 'month')) {
      list.push(curr);
      curr = curr.add(1, 'month');
    }
    return list;
  }, [calendar]);

  const occupancyDays = useMemo(() => {
    const startOfMonth = viewDate.startOf('month');
    const endOfMonth = viewDate.endOf('month');
    const days = [];
    let hasAddedPadding = false;

    const studentAllocations = isDynamic 
        ? students.map(s => {
            const fullMatrice = [...matrice, ...(s.customCompetencies || [])];
            const plan = calculateDynamicPlanning(s, calendar, fullMatrice);
            const dailyMaps = {};
            Object.entries(plan.allocationsByDate).forEach(([dateStr, allocs]) => {
                let hoursInRoom = 0;
                allocs.forEach(alloc => {
                    const comp = fullMatrice.find(c => c.code === alloc.code);
                    const skillStat = plan.skillStats.find(st => st.code === alloc.code);
                    const status = skillStat?.status || 'upcoming';

                    // Filtre de statut
                    if (statusFilter !== 'all' && status !== statusFilter) return;

                    if (comp && comp.local === selectedRoom) hoursInRoom += alloc.hours;
                });
                if (hoursInRoom > 0) dailyMaps[dateStr] = hoursInRoom;
            });
            return { student: s, dailyMaps };
        }) 
        : [];

    for (let d = 1; d <= endOfMonth.date(); d++) {
      const date = startOfMonth.date(d), dayOfWeek = date.day(); 
      if (dayOfWeek === 0 || dayOfWeek === 6) continue;
      if (!hasAddedPadding) {
        for (let p = 0; p < dayOfWeek - 1; p++) days.push({ padding: true });
        hasAddedPadding = true;
      }
      const calendarDay = calendar.find(calDate => calDate.date.isSame(date, 'day'));
      let presentStudents = [];
      let totalStudentHours = 0;

      if (calendarDay) {
        const dateStr = date.format('YYYY-MM-DD');
        if (isDynamic) {
             studentAllocations.forEach(({ student, dailyMaps }) => {
                 const h = dailyMaps[dateStr];
                 if (h > 0) { presentStudents.push({ name: student.name, hours: h }); totalStudentHours += h; }
             });
        } else {
            students.forEach(student => {
                const fullMatrice = [...matrice, ...(student.customCompetencies || [])];
                const { firstWorkDayIndex } = calculateStudentProgress(student, calendar);
                if (firstWorkDayIndex !== -1) {
                    const absoluteIndex = calendar.findIndex(calDate => calDate.date.isSame(date, 'day'));
                    const studentDayNum = absoluteIndex - firstWorkDayIndex;
                    if (studentDayNum >= 0) {
                        const week = Math.floor(studentDayNum / 5);
                        const totalWeeklyHours = fullMatrice.reduce((sum, c) => sum + (c.weeklyHours[week] || 0), 0);
                        const roomSkill = fullMatrice.find(m => m.local === selectedRoom && (m.weeklyHours[week] || 0) > 0);
                        if (roomSkill && totalWeeklyHours > 0) {
                            const h = (roomSkill.weeklyHours[week] / totalWeeklyHours) * (calendarDay.hours || 6);
                            presentStudents.push({ name: student.name, hours: h });
                            totalStudentHours += h;
                        }
                    }
                }
            });
        }
      }
      days.push({ date, padding: false, presentStudents, totalHours: totalStudentHours, isWorkable: !!calendarDay });
    }
    return days;
  }, [selectedRoom, viewDate, students, calendar, matrice, isDynamic, statusFilter]);

  const selectedDayData = useMemo(() => 
    occupancyDays.find(d => !d.padding && d.date.format('YYYY-MM-DD') === selectedDayStr), 
    [occupancyDays, selectedDayStr]
  );

  return (
    <div className="d-flex flex-column h-100 bg-white">
      <div className="d-flex justify-content-between align-items-end mb-4 d-print-none gap-3">
        <div className="flex-grow-1 row g-2">
            <div className="col-md-6">
                <label className="small fw-bold text-muted text-uppercase mb-1 d-block">Local</label>
                <select value={selectedRoom} onChange={(e) => setSelectedRoom(e.target.value)} className="form-select form-select-sm fw-bold">
                    {rooms.map(r => <option key={r} value={r}>{r}</option>)}
                </select>
            </div>
            <div className="col-md-6">
                <label className="small fw-bold text-muted text-uppercase mb-1 d-block">Mois</label>
                <select value={viewDate.format('YYYY-MM')} onChange={(e) => setViewDate(dayjs(e.target.value))} className="form-select form-select-sm">
                    {monthsList.map(m => <option key={m.format('YYYY-MM')} value={m.format('YYYY-MM')}>{m.format('MMMM YYYY').toUpperCase()}</option>)}
                </select>
            </div>
        </div>
        <div className="text-end">
            <label className="small fw-bold text-muted text-uppercase mb-1 d-block">Filtre</label>
            <div className="btn-group btn-group-sm shadow-sm">
                <button onClick={()=>setStatusFilter('all')} className={`btn ${statusFilter==='all'?'btn-dark':'btn-outline-dark'}`}>Tout</button>
                <button onClick={()=>setStatusFilter('in_progress')} className={`btn ${statusFilter==='in_progress'?'btn-primary':'btn-outline-primary'}`}>En cours</button>
                <button onClick={()=>setStatusFilter('upcoming')} className={`btn ${statusFilter==='upcoming'?'btn-secondary':'btn-outline-secondary'}`}>À venir</button>
            </div>
        </div>
      </div>

      <div className="row flex-grow-1 overflow-hidden">
        <div className="col-md-8 h-100 overflow-auto">
            <div className="d-grid gap-1 bg-light border rounded overflow-hidden" style={{ gridTemplateColumns: 'repeat(5, 1fr)' }}>
                {['Lun', 'Mar', 'Mer', 'Jeu', 'Ven'].map(d => <div key={d} className="bg-white p-2 text-center small fw-bold text-secondary border-bottom">{d}</div>)}
                {occupancyDays.map((day, i) => {
                    const dateStr = day.date?.format('YYYY-MM-DD');
                    const isSelected = selectedDayStr === dateStr;
                    const tooltipText = day.presentStudents?.map(s => `${s.name} (${s.hours}h)`).join('<br/>') || 'Vide';
                    return (
                        <div key={i} onClick={() => !day.padding && day.isWorkable && setSelectedDayStr(dateStr)} className={`p-2 bg-white cursor-pointer ${day.padding ? 'bg-light opacity-25' : ''} ${!day.padding && !day.isWorkable ? 'bg-light' : 'hover-bg-light'} ${isSelected ? 'border border-primary border-2' : ''}`} style={{ minHeight: '110px' }}>
                            {!day.padding && (
                            <>
                                <div className="text-end small fw-bold mb-1">{day.date.date()}</div>
                                {day.isWorkable && day.presentStudents.length > 0 ? (
                                    <div className="d-flex flex-column align-items-center justify-content-center h-100" data-bs-toggle="tooltip" data-bs-html="true" title={tooltipText}>
                                        <span className="h2 fw-black text-primary mb-0">{day.presentStudents.length}</span>
                                        <div className="badge bg-info bg-opacity-10 text-info border border-info border-opacity-25 rounded-pill" style={{fontSize:'0.6rem'}}>{day.totalHours.toFixed(0)}h</div>
                                    </div>
                                ) : day.isWorkable && <div className="text-center opacity-25 mt-3 small fw-bold">LIBRE</div>}
                            </>
                            )}
                        </div>
                    );
                })}
            </div>
        </div>
        <div className="col-md-4">
            <div className="card border-0 shadow-sm bg-light h-100">
                <div className="card-body">
                    <h6 className="fw-bold text-dark border-bottom pb-2 small text-uppercase">Détail : {selectedDayData ? selectedDayData.date.format('D MMM YYYY') : 'Sélection'}</h6>
                    {selectedDayData && selectedDayData.presentStudents.length > 0 ? (
                        <ul className="list-group list-group-flush rounded border mt-2 shadow-sm">
                            {selectedDayData.presentStudents.map((s, idx) => (
                                <li key={idx} className="list-group-item d-flex justify-content-between py-2 small">
                                    <span className="fw-bold">{s.name}</span><span className="badge bg-primary">{s.hours}h</span>
                                </li>
                            ))}
                        </ul>
                    ) : <div className="text-center py-5 text-muted small">Aucune présence avec ces filtres</div>}
                </div>
            </div>
        </div>
      </div>
    </div>
  );
};

window.RoomOccupancy = RoomOccupancy;