const MonthlySchedule = ({ student, calendar, matrice, isDynamic, onUpdateStudent }) => {
  const { useState, useMemo, useEffect } = React;
  
  const [viewDate, setViewDate] = useState(dayjs().startOf('month'));
  const [selectedDay, setSelectedDay] = useState(null);
  const [statusFilter, setStatusFilter] = useState('all'); // 'all' | 'in_progress' | 'upcoming'
  const [eventType, setEventType] = useState('exam'); 
  const [targetCompCode, setTargetCompCode] = useState('');
  const [eventDuration, setEventDuration] = useState(3);
  const [eventComment, setEventComment] = useState('');

  const fullMatrice = useMemo(() => [...matrice, ...(student?.customCompetencies || [])], [matrice, student]);

  const monthsList = useMemo(() => {
    if (calendar.length === 0) return [];
    const start = calendar[0].date.startOf('month'), end = calendar[calendar.length - 1].date.startOf('month'), list = [];
    let curr = start;
    while(curr.isBefore(end) || curr.isSame(end, 'month')) { list.push(curr); curr = curr.add(1, 'month'); }
    return list;
  }, [calendar]);

  const planning = useMemo(() => isDynamic && student ? window.calculateDynamicPlanning(student, calendar, fullMatrice) : null, [isDynamic, student, calendar, fullMatrice]);

  useEffect(() => {
    if (window.bootstrap) {
      const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
      tooltipTriggerList.map(el => new bootstrap.Tooltip(el));
    }
    return () => {
      const tooltips = document.querySelectorAll('.tooltip');
      tooltips.forEach(t => t.remove());
    };
  }, [viewDate, student, planning, statusFilter]);

  const scheduleDays = useMemo(() => {
    if (!student) return [];
    const startOfMonth = viewDate.startOf('month'), endOfMonth = viewDate.endOf('month'), days = [];
    let hasPadding = false;

    for (let d = 1; d <= endOfMonth.date(); d++) {
      const date = startOfMonth.date(d), dayOfWeek = date.day();
      if (dayOfWeek === 0 || dayOfWeek === 6) continue;
      if (!hasPadding) { for (let p = 0; p < dayOfWeek - 1; p++) days.push({ padding: true }); hasPadding = true; }

      const calDay = calendar.find(c => c.date.isSame(date, 'day'));
      let competencies = [];
      let hours = 0;

      if (calDay) {
        hours = window.getEffectiveDailyHours(student, calDay);
        if (hours > 0 || isDynamic) {
            const dateStr = date.format('YYYY-MM-DD');
            const dayAllocs = (planning?.allocationsByDate[dateStr] || []);
            dayAllocs.forEach(alloc => {
                const comp = fullMatrice.find(c => c.code === alloc.code);
                const skillStat = planning?.skillStats.find(s => s.code === alloc.code);
                const status = skillStat?.status || 'upcoming';

                if (statusFilter !== 'all' && status !== statusFilter) return;

                let eventComment = "";
                if (alloc.isExam) {
                   eventComment = (student.examEvents || []).find(e => e.date === dateStr && e.competencyCode === alloc.code)?.comment || "";
                } else if (alloc.isInternship) {
                   eventComment = (student.internshipEvents || []).find(e => e.date === dateStr && e.competencyCode === alloc.code)?.comment || "";
                }

                if (comp) competencies.push({ 
                    ...comp, 
                    dailyH: alloc.hours, 
                    isExam: !!alloc.isExam,
                    isInternship: !!alloc.isInternship,
                    status: status,
                    comment: eventComment
                });
            });
        }
      }
      days.push({ padding: false, date, isWorkable: !!calDay, competencies, hours });
    }
    return days;
  }, [student, viewDate, calendar, fullMatrice, isDynamic, planning, statusFilter]);

  const handleAddEvent = (e) => {
    e.preventDefault();
    if (!selectedDay || !targetCompCode) return;
    if (eventType === 'exam') {
        const newExam = { id: crypto.randomUUID(), competencyCode: targetCompCode, date: selectedDay, duration: eventDuration, comment: eventComment };
        onUpdateStudent({ ...student, examEvents: [...(student.examEvents || []), newExam] });
    } else {
        const newIntern = { id: crypto.randomUUID(), competencyCode: targetCompCode, date: selectedDay, duration: eventDuration, comment: eventComment };
        onUpdateStudent({ ...student, internshipEvents: [...(student.internshipEvents || []), newIntern] });
    }
    setTargetCompCode('');
    setEventComment('');
  };

  const handleRemoveEvent = (id, type) => {
    if (type === 'exam') {
        onUpdateStudent({ ...student, examEvents: (student.examEvents || []).filter(e => e.id !== id) });
    } else {
        onUpdateStudent({ ...student, internshipEvents: (student.internshipEvents || []).filter(e => e.id !== id) });
    }
  };

  const selectedDayCapacity = useMemo(() => {
      if (!selectedDay) return 6;
      const d = calendar.find(c => c.date.format('YYYY-MM-DD') === selectedDay);
      return d ? d.hours : 6;
  }, [selectedDay, calendar]);

  const eventsOnSelectedDay = useMemo(() => {
      if (!selectedDay || !student) return [];
      const exams = (student.examEvents || []).filter(e => e.date === selectedDay).map(e => ({ ...e, type: 'exam' }));
      const interns = (student.internshipEvents || []).filter(e => e.date === selectedDay).map(e => ({ ...e, type: 'internship' }));
      return [...exams, ...interns];
  }, [selectedDay, student]);

  if (!student) return <div className="p-5 text-center text-muted border rounded bg-light m-4">Sélectionnez un étudiant.</div>;

  return (
    <div className="d-flex flex-column h-100 position-relative bg-white overflow-hidden">
      <div className="d-flex justify-content-between align-items-end mb-4 d-print-none px-1">
        <div className="vstack gap-2">
            <h2 className="h4 fw-bold text-dark d-flex align-items-center gap-2 mb-0">
                Horaire : {viewDate.format('MMMM YYYY').toUpperCase()}
                <span className={`badge ${isDynamic ? 'bg-primary' : 'bg-secondary'} bg-opacity-10 ${isDynamic ? 'text-primary' : 'text-secondary'} border small`}>{isDynamic ? 'Dynamique' : 'Statique'}</span>
            </h2>
            <div className="d-flex align-items-center gap-2">
                <span className="small fw-bold text-muted text-uppercase" style={{fontSize:'0.65rem'}}>Filtre :</span>
                <div className="btn-group btn-group-sm">
                    <button onClick={()=>setStatusFilter('all')} className={`btn ${statusFilter==='all'?'btn-dark':'btn-outline-dark'}`}>Tout</button>
                    <button onClick={()=>setStatusFilter('in_progress')} className={`btn ${statusFilter==='in_progress'?'btn-primary':'btn-outline-primary'}`}>En cours</button>
                    <button onClick={()=>setStatusFilter('upcoming')} className={`btn ${statusFilter==='upcoming'?'btn-secondary':'btn-outline-secondary'}`}>À venir</button>
                </div>
            </div>
        </div>
        <select value={viewDate.format('YYYY-MM')} onChange={e => setViewDate(dayjs(e.target.value))} className="form-select form-select-sm w-auto shadow-sm">
          {monthsList.map(m => <option key={m.format('YYYY-MM')} value={m.format('YYYY-MM')}>{m.format('MMMM YYYY').toUpperCase()}</option>)}
        </select>
      </div>

      <div className="d-flex border-top border-start bg-light shadow-sm">
        {['Lun', 'Mar', 'Mer', 'Jeu', 'Ven'].map(d => <div key={d} style={{ width: '20%' }} className="py-2 text-center small fw-bold text-uppercase text-secondary border-bottom border-end">{d}</div>)}
      </div>

      <div className="flex-grow-1 border-start d-grid" style={{ gridTemplateColumns: 'repeat(5, 1fr)', gridAutoRows: 'minmax(140px, 1fr)', overflowY: 'auto' }}>
        {scheduleDays.map((day, i) => (
            <div key={i} onClick={() => !day.padding && day.isWorkable && setSelectedDay(day.date.format('YYYY-MM-DD'))} 
                className={`position-relative border-end border-bottom p-2 d-flex flex-column cursor-pointer transition-all hover-bg-light ${day.padding ? 'bg-light opacity-25' : ''}`}
                style={{ backgroundColor: day.padding ? '#f8f9fa' : '#ffffff' }}>
                {!day.padding && (
                    <>
                        <div className="d-flex justify-content-between mb-1">
                            <span className="fs-5 fw-bold text-dark">{day.date.date()}</span>
                            {day.hours > 0 && <span className="badge bg-light text-muted border fw-normal" style={{fontSize:'0.6rem'}}>{day.hours}h école</span>}
                        </div>
                        <div className="vstack gap-1">
                            {day.competencies.map((c, j) => {
                                let bc = c.isExam ? '#0d6efd' : c.isInternship ? '#6f42c1' : '#adb5bd';
                                const isUpcoming = c.status === 'upcoming';
                                const tooltipTitle = `${c.name} (${isUpcoming ? 'À venir' : 'En cours'})${c.comment ? `\nNote : ${c.comment}` : ''}`;
                                return (
                                    <div key={j} className={`rounded p-1 shadow-sm border ${isUpcoming ? 'border-dashed opacity-75' : 'border-3'} text-truncate`} data-bs-toggle="tooltip" title={tooltipTitle}
                                        style={{fontSize:'0.7rem', borderColor: bc, backgroundColor: c.isExam ? '#cfe2ff' : c.isInternship ? '#f3e8ff' : '#fff', color: '#000'}}>
                                        <div className="fw-bold d-flex justify-content-between">
                                            <span>{c.isExam?'🎓':c.isInternship?'🏢':''} C-{c.number}</span><span className={isUpcoming ? 'opacity-75 fst-italic' : 'text-primary'}>{c.dailyH}h</span>
                                        </div>
                                    </div>
                                );
                            })}
                        </div>
                    </>
                )}
            </div>
        ))}
      </div>

      {selectedDay && (
          <div className="position-absolute top-0 end-0 h-100 bg-white shadow-lg border-start d-print-none" style={{ width: '340px', zIndex: 1050 }}>
              <div className="p-4 vstack gap-4 h-100 overflow-auto">
                  <div className="d-flex justify-content-between align-items-center border-bottom pb-2">
                      <h5 className="fw-bold mb-0 text-dark">Le {dayjs(selectedDay).format('D MMMM')}</h5>
                      <button onClick={() => setSelectedDay(null)} className="btn-close"></button>
                  </div>

                  <div>
                      <h6 className="small fw-black text-secondary text-uppercase mb-3" style={{fontSize:'0.65rem'}}>Événements existants</h6>
                      {eventsOnSelectedDay.length > 0 ? (
                        <div className="vstack gap-2">
                            {eventsOnSelectedDay.map(ev => (
                                <div key={ev.id} className="p-2 border rounded bg-light d-flex flex-column gap-1 shadow-sm">
                                    <div className="d-flex justify-content-between align-items-center">
                                        <div className="small fw-bold">
                                            <span className={`badge ${ev.type==='exam'?'bg-primary':'bg-purple'} me-1`} style={ev.type==='internship'?{backgroundColor:'#6f42c1'}:{}}>{ev.type==='exam'?'ÉPREUVE':'STAGE'}</span>
                                            C-{fullMatrice.find(m=>m.code===ev.competencyCode)?.number} ({ev.duration}h)
                                        </div>
                                        <button onClick={()=>handleRemoveEvent(ev.id, ev.type)} className="btn btn-sm btn-link text-danger p-0 shadow-none"><i className="bi bi-trash3-fill"></i></button>
                                    </div>
                                    {ev.comment && <div className="small text-muted fst-italic border-top pt-1 mt-1">{ev.comment}</div>}
                                </div>
                            ))}
                        </div>
                      ) : <p className="text-center small text-muted py-2 fst-italic">Rien au programme.</p>}
                  </div>

                  <div className="card bg-light border-0 shadow-sm mt-auto">
                    <div className="card-body">
                      <h6 className="small fw-black text-dark text-uppercase mb-3" style={{fontSize:'0.65rem'}}>Nouvel événement</h6>
                      <div className="btn-group btn-group-sm w-100 mb-3">
                          <button onClick={()=>setEventType('exam')} className={`btn ${eventType==='exam'?'btn-primary':'btn-outline-primary'} fw-bold`}>ÉPREUVE</button>
                          <button onClick={()=>setEventType('internship')} className={`btn ${eventType==='internship'?'btn-purple text-white':'btn-outline-purple'} fw-bold`} style={eventType==='internship'?{backgroundColor:'#6f42c1'}:{}}>STAGE</button>
                      </div>
                      <form onSubmit={handleAddEvent} className="vstack gap-3">
                          <select className="form-select form-select-sm" value={targetCompCode} onChange={e => setTargetCompCode(e.target.value)} required><option value="">Compétence...</option>{fullMatrice.map(m => <option key={m.code} value={m.code}>C-{m.number} : {m.name}</option>)}</select>
                          <select className="form-select form-select-sm" value={eventDuration} onChange={e => setEventDuration(parseInt(e.target.value))} required>
                              {Array.from({length: eventType === 'internship' ? 10 : selectedDayCapacity}).map((_,i)=><option key={i+1} value={i+1}>{i+1} heure{i>0?'s':''}</option>)}
                          </select>
                          <input type="text" className="form-control form-control-sm" placeholder="Commentaire..." value={eventComment} onChange={e => setEventComment(e.target.value)} />
                          <button type="submit" className="btn btn-dark btn-sm fw-bold w-100 mt-2 shadow-sm text-white"><i className="bi bi-plus-lg me-1"></i> Confirmer</button>
                      </form>
                    </div>
                  </div>
              </div>
          </div>
      )}
      <style>{`
        .border-dashed { border-style: dashed !important; }
        .border-3 { border-width: 3px !important; }
      `}</style>
    </div>
  );
};
window.MonthlySchedule = MonthlySchedule;