const ROW_HEIGHT = '48px';
const STICKY_COL_WIDTH = '360px';
const GRID_START_PADDING = '50px';
const COLORS = { 'Théorique': '#0d6efd', 'Pratique': '#198754', 'Stage': '#6f42c1', 'Static': '#adb5bd' };

const MatrixViewer = ({ matrice, onUpdateMatrice, student, calendar, isDynamic = false }) => {
  const { useRef, useMemo, useEffect } = React;
  const { calculateDynamicPlanning, calculateProjectedPlanning } = window;
  const scrollContainerRef = useRef(null);

  // Initialisation des tooltips Bootstrap
  useEffect(() => {
    if (window.bootstrap) {
      const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
      tooltipTriggerList.map(el => new bootstrap.Tooltip(el, { container: 'body' }));
    }
    return () => {
      const tooltips = document.querySelectorAll('.tooltip');
      tooltips.forEach(t => t.remove());
    };
  }, [student, matrice, calendar, isDynamic]);

  const handleMouseDown = (e) => {
    if (!scrollContainerRef.current) return;
    scrollContainerRef.current.classList.add('cursor-grabbing');
    scrollContainerRef.current.startX = e.pageX;
    scrollContainerRef.current.scrollLeftStart = scrollContainerRef.current.scrollLeft;
  };
  const handleMouseMove = (e) => {
    if (!scrollContainerRef.current?.classList.contains('cursor-grabbing')) return;
    const x = e.pageX, walk = (x - scrollContainerRef.current.startX) * 2;
    scrollContainerRef.current.scrollLeft = scrollContainerRef.current.scrollLeftStart - walk;
  };
  const handleMouseUpOrLeave = () => { scrollContainerRef.current?.classList.remove('cursor-grabbing'); };

  const totalWorkingDays = calendar.length;
  const monthSections = useMemo(() => {
      if (totalWorkingDays === 0) return [];
      const sections = []; let currentMonth = -1, startIndex = 0;
      calendar.forEach((day, index) => {
          const m = day.date.month();
          if (m !== currentMonth) {
              if (currentMonth !== -1) sections.push({ label: calendar[startIndex].date.format('MMM YY'), left: (startIndex / totalWorkingDays) * 100, width: ((index - startIndex) / totalWorkingDays) * 100 });
              currentMonth = m; startIndex = index;
          }
      });
      if (startIndex < totalWorkingDays) sections.push({ label: calendar[startIndex].date.format('MMM YY'), left: (startIndex / totalWorkingDays) * 100, width: ((totalWorkingDays - startIndex) / totalWorkingDays) * 100 });
      return sections;
  }, [calendar, totalWorkingDays]);

  const fullMatrice = useMemo(() => student ? [...matrice, ...(student.customCompetencies || [])] : matrice, [matrice, student]);
  const studentPlanning = useMemo(() => student && calendar.length > 0 ? calculateDynamicPlanning(student, calendar, fullMatrice) : null, [student, calendar, fullMatrice]);
  const projectedPlanning = useMemo(() => student && calendar.length > 0 ? calculateProjectedPlanning(student, calendar, fullMatrice) : null, [student, calendar, fullMatrice]);

  const getPosition = (date) => {
      if (totalWorkingDays === 0 || !date) return 0;
      let idx = calendar.findIndex(c => c.date.isSame(date, 'day'));
      if (idx === -1) {
          if (date.isBefore(calendar[0].date)) return 0;
          if (date.isAfter(calendar[totalWorkingDays-1].date)) return 100;
          return (calendar.findIndex(c => c.date.isAfter(date)) / totalWorkingDays) * 100;
      }
      return (idx / totalWorkingDays) * 100;
  };
  const getWidth = (start, end) => {
      if (totalWorkingDays === 0 || !start || !end) return 0;
      const startIdx = Math.max(0, calendar.findIndex(c => c.date.isSame(start, 'day') || c.date.isAfter(start)));
      const endIdx = calendar.findIndex(c => c.date.isSame(end, 'day') || c.date.isAfter(end));
      if (endIdx === -1) return 0;
      const diff = Math.max(1, endIdx - startIdx + 1);
      return (diff / totalWorkingDays) * 100;
  };

  return (
    <div className="d-flex flex-column h-100">
      <div className="mb-3 d-flex justify-content-between align-items-center no-print px-1">
        <div><h2 className="h5 fw-bold mb-0">Ligne du temps : {student?.name || 'Programme'}</h2></div>
      </div>
      <div ref={scrollContainerRef} className="flex-grow-1 overflow-auto border rounded-3 bg-white shadow-sm cursor-grab" onMouseDown={handleMouseDown} onMouseMove={handleMouseMove} onMouseUp={handleMouseUpOrLeave} onMouseLeave={handleMouseUpOrLeave}>
        <div style={{ minWidth: '2400px', minHeight: '100%' }} className="position-relative">
          <div className="d-flex bg-light border-bottom sticky-top" style={{ height: '36px', zIndex: 1000 }}>
            <div className="border-end p-2 fw-bold small text-secondary text-uppercase sticky-left bg-light shadow-sm d-flex align-items-center flex-shrink-0" style={{ width: STICKY_COL_WIDTH, minWidth: STICKY_COL_WIDTH, position: 'sticky', left: 0, zIndex: 1100 }}>Compétences</div>
            <div className="flex-grow-1 position-relative" style={{ marginLeft: GRID_START_PADDING }}>
                {monthSections.map((section, idx) => (<div key={idx} className="position-absolute top-0 bottom-0 border-start px-1 small fw-black text-secondary text-uppercase d-flex align-items-center overflow-hidden" style={{ left: `${section.left}%`, width: `${section.width}%`, fontSize: '0.65rem' }}>{section.label}</div>))}
            </div>
          </div>
          <div className="position-relative">
            {fullMatrice.map((comp) => {
              const projStat = projectedPlanning?.skillStats.find(s => s.code === comp.code);
              const dynStat = studentPlanning?.skillStats.find(s => s.code === comp.code);
              const evaluation = student?.evaluations?.[comp.code];
              
              let projLeft = 0, projWidth = 0;
              if (projStat?.projectedStart && projStat?.projectedEnd) { projLeft = getPosition(projStat.projectedStart); projWidth = getWidth(projStat.projectedStart, projStat.projectedEnd); }
              
              let dynLeft = 0, dynWidth = 0;
              let dynEndStr = '--', activityEndStr = '--';
              if (isDynamic && dynStat?.dynamicStart && dynStat?.dynamicEnd && dynStat.status !== 'upcoming') { 
                  dynLeft = getPosition(dynStat.dynamicStart); 
                  dynWidth = getWidth(dynStat.dynamicStart, dynStat.dynamicEnd); 
                  dynEndStr = dynStat.dynamicEnd.format('D MMM YYYY');
                  if (dynStat.activityEnd) activityEndStr = dynStat.activityEnd.format('D MMM YYYY');
              }

              const typeColor = COLORS[comp.type] || COLORS['Théorique'];
              const resultStr = evaluation?.result === 'success' ? 'RÉUSSITE' : (evaluation?.result === 'failure' ? 'ÉCHEC' : 'En cours');
              const resultDateStr = evaluation?.resultDate ? dayjs(evaluation.resultDate).format('D MMM YYYY') : '--';

              const theoryTooltip = `
                <div class='text-start small p-1'>
                    <div class='fw-bold border-bottom pb-1 mb-1 text-uppercase' style='font-size:0.6rem'>SELON LA MATRICE</div>
                    <div class='fw-bold mb-1'>${comp.name}</div>
                    <div><b>Quota :</b> ${comp.hours}h</div>
                    <div class='pt-1 border-top opacity-75'>Période : du ${projStat?.projectedStart ? projStat.projectedStart.format('D MMM') : '--'} au ${projStat?.projectedEnd ? projStat.projectedEnd.format('D MMM') : '--'}</div>
                </div>
              `;
              
              const realTooltipHtml = `
                <div class='text-start small p-1'>
                    <div class='fw-bold border-bottom pb-1 mb-1 text-uppercase' style='font-size:0.6rem'>Planification Réelle</div>
                    <div class='fw-bold mb-1'>${comp.name}</div>
                    <div><b>Début :</b> ${dynStat?.dynamicStart ? dynStat.dynamicStart.format('D MMM YYYY') : '--'}</div>
                    <div style='color: #0dcaf0'><b>Fin cours :</b> ${dynEndStr}</div>
                    <div style='color: #d0bfff'><b>Fin compétence :</b> ${activityEndStr}</div>
                    <div class='mt-1 pt-1 border-top'>Verdict : ${resultStr}</div>
                </div>
              `;

              return (
                <div key={comp.code} className="d-flex border-bottom hover-bg-light" style={{ height: ROW_HEIGHT }}>
                  <div className="border-end px-3 sticky-left shadow-sm flex-shrink-0 d-flex align-items-center" style={{ width: STICKY_COL_WIDTH, minWidth: STICKY_COL_WIDTH, position: 'sticky', left: 0, zIndex: 900, backgroundColor: 'white' }}>
                    <div className="text-truncate fw-bold text-dark w-100 d-flex align-items-center gap-1" style={{ fontSize: '0.75rem' }}><span className="badge border me-1 shadow-sm" style={{ backgroundColor: typeColor, color: '#fff', borderColor: typeColor }}>C{comp.number}</span><span className="text-truncate">{comp.name}</span></div>
                  </div>
                  <div className="flex-grow-1 position-relative" style={{ marginLeft: GRID_START_PADDING, zIndex: 10 }}>
                      {projWidth > 0 && (
                        <div className="position-absolute rounded shadow-sm text-white fw-bold d-flex align-items-center justify-content-center cursor-help" 
                            data-bs-toggle="tooltip" data-bs-html="true" 
                            title={theoryTooltip}
                            style={{ top: isDynamic ? '8px' : '10px', bottom: isDynamic ? '8px' : '10px', left: `${projLeft}%`, width: `${projWidth}%`, fontSize: '0.65rem', backgroundColor: isDynamic ? COLORS['Static'] : typeColor, opacity: isDynamic ? 0.4 : 1, zIndex: 1 }}>
                            {!isDynamic && `${comp.hours}h`}
                        </div>
                      )}
                      {isDynamic && dynWidth > 0 && (
                        <div className="position-absolute rounded shadow-sm text-white fw-bold d-flex align-items-center justify-content-center border border-white border-1 cursor-help" 
                            data-bs-toggle="tooltip" data-bs-html="true" 
                            title={realTooltipHtml}
                            style={{ top: '15px', bottom: '15px', left: `${dynLeft}%`, width: `${dynWidth}%`, fontSize: '0.65rem', backgroundColor: typeColor, zIndex: 2 }}>
                            {comp.hours}h
                        </div>
                      )}
                  </div>
                </div>
              );
            })}
          </div>
        </div>
      </div>
    </div>
  );
};
window.MatrixViewer = MatrixViewer;