
const MatrixEditor = ({ matrice, onUpdateMatrice }) => {
  const { useState, useMemo, useRef } = React;
  
  const WEEKS_COUNT = 60;
  const WEEK_COL_WIDTH = 50;
  const TARGET_HOURS_PER_WEEK = 30;
  const STICKY_COL_WIDTH = '400px';
  const COLORS = { 'Théorique': '#0d6efd', 'Pratique': '#198754', 'Stage': '#6f42c1' };
  const HOUR_OPTIONS = [0, 3, 6, 9, 12, 15, 18, 21, 24, 27, 30];

  const [editingCompCode, setEditingCompCode] = useState(null);
  const fileInputRef = useRef(null);

  const editingComp = useMemo(() => matrice.find(c => c.code === editingCompCode), [matrice, editingCompCode]);
  const totalMatrixHours = useMemo(() => matrice.reduce((sum, comp) => sum + (comp.hours || 0), 0), [matrice]);
  
  const weeklyTotals = useMemo(() => {
    const totals = Array(WEEKS_COUNT).fill(0);
    matrice.forEach(comp => comp.weeklyHours.forEach((h, i) => { if (i < WEEKS_COUNT) totals[i] += h; }));
    return totals;
  }, [matrice]);

  const updateComp = (code, updates) => {
    onUpdateMatrice(matrice.map(c => {
      if (c.code === code) {
        const updated = { ...c, ...updates };
        if (updates.weeklyHours) { updated.hours = updated.weeklyHours.reduce((a, b) => a + b, 0); updated.dailyHours = Array(updated.hours).fill(1); }
        return updated;
      }
      return c;
    }));
  };

  const moveComp = (code, delta) => {
    const comp = matrice.find(c => c.code === code);
    if (!comp) return;
    const duration = comp.endWeek - comp.startWeek;
    const newStart = Math.max(0, Math.min(WEEKS_COUNT - 1 - duration, comp.startWeek + delta));
    const newWeekly = Array(WEEKS_COUNT).fill(0);
    for (let i = 0; i <= duration; i++) newWeekly[newStart + i] = comp.weeklyHours[comp.startWeek + i] || 0;
    updateComp(code, { startWeek: newStart, endWeek: newStart + duration, weeklyHours: newWeekly });
  };

  const resizeComp = (code, side, delta) => {
    const comp = matrice.find(c => c.code === code);
    if (!comp) return;
    let newStart = comp.startWeek, newEnd = comp.endWeek, newWeekly = [...comp.weeklyHours];
    if (side === 'start') {
      const target = comp.startWeek + delta;
      if (target >= 0 && target <= comp.endWeek) { newStart = target; if (delta < 0) newWeekly[newStart] = 15; if (delta > 0) newWeekly[comp.startWeek] = 0; }
    } else {
      const target = comp.endWeek + delta;
      if (target >= comp.startWeek && target < WEEKS_COUNT) { newEnd = target; if (delta > 0) newWeekly[newEnd] = 15; if (delta < 0) newWeekly[comp.endWeek] = 0; }
    }
    updateComp(code, { startWeek: newStart, endWeek: newEnd, weeklyHours: newWeekly });
  };

  const splitComp = (code, weekIndex) => {
    const comp = matrice.find(c => c.code === code);
    if (!comp || weekIndex >= comp.endWeek) return;

    const p1Weekly = [...comp.weeklyHours].map((h, i) => i <= weekIndex ? h : 0);
    const p2Weekly = [...comp.weeklyHours].map((h, i) => i > weekIndex ? h : 0);

    const c1 = { 
        ...comp, 
        weeklyHours: p1Weekly, 
        endWeek: weekIndex, 
        hours: p1Weekly.reduce((a, b) => a + b, 0),
        dailyHours: Array(p1Weekly.reduce((a, b) => a + b, 0)).fill(1)
    };

    const c2 = {
        ...comp,
        code: 'C-' + Date.now(),
        name: comp.name + ' (Partie 2)',
        weeklyHours: p2Weekly,
        startWeek: weekIndex + 1,
        hours: p2Weekly.reduce((a, b) => a + b, 0),
        dailyHours: Array(p2Weekly.reduce((a, b) => a + b, 0)).fill(1)
    };

    const nextMatrice = [];
    matrice.forEach(c => {
        if (c.code === code) { nextMatrice.push(c1); nextMatrice.push(c2); }
        else { nextMatrice.push(c); }
    });
    onUpdateMatrice(nextMatrice);
  };

  return (
    <div className="d-flex flex-column h-100 position-relative overflow-hidden">
      <div className="d-flex justify-content-between align-items-center mb-4 px-1">
        <div className="d-flex align-items-center gap-3">
          <h2 className="h4 fw-bold mb-0 text-dark">Éditeur de matrice</h2>
          <div className="badge bg-dark rounded-pill px-3 py-2 shadow-sm">Total : {totalMatrixHours}h</div>
        </div>
        <button onClick={() => {
            const nextNum = matrice.length > 0 ? Math.max(...matrice.map(c => c.number)) + 1 : 1;
            onUpdateMatrice([...matrice, { number: nextNum, code: 'C-'+Date.now(), name: 'Nouveau', hours: 30, type: 'Théorique', local: 'Salle', weeklyHours: Array(WEEKS_COUNT).fill(0).map((_,i)=>i<2?15:0), startWeek: 0, endWeek: 1 }]);
        }} className="btn btn-sm btn-primary shadow-sm">+ Ajouter une ligne</button>
      </div>

      <div className="flex-grow-1 overflow-auto border rounded bg-white shadow-sm position-relative">
        <div style={{ minWidth: `calc(${STICKY_COL_WIDTH} + ${WEEKS_COUNT * WEEK_COL_WIDTH}px)` }}>
          <div className="d-flex bg-light border-bottom sticky-top" style={{ height: '48px', zIndex: 1000 }}>
            <div className="p-2 fw-bold small text-secondary border-end sticky-left bg-light d-flex align-items-center shadow-sm" 
                 style={{ width: STICKY_COL_WIDTH, minWidth: STICKY_COL_WIDTH, left: 0, zIndex: 1100, position: 'sticky' }}>
                 Compétence
            </div>
            <div className="d-flex">
                {Array.from({ length: WEEKS_COUNT }).map((_, i) => (
                    <div key={i} className={`text-center border-end small fw-bold d-flex flex-column justify-content-center ${weeklyTotals[i] > TARGET_HOURS_PER_WEEK ? 'bg-danger text-white' : 'text-muted'}`} style={{ width: WEEK_COL_WIDTH+'px' }}>
                        <div style={{fontSize:'0.6rem'}}>S{i+1}</div><div style={{fontSize:'0.6rem'}}>{weeklyTotals[i]}h</div>
                    </div>
                ))}
            </div>
          </div>
          {matrice.map((comp, idx) => (
            <div key={comp.code} className="d-flex border-bottom hover-bg-light align-items-center" style={{ minHeight: '80px' }}>
                <div className="p-2 border-end sticky-left bg-white d-flex gap-2 align-items-center shadow-sm" 
                     style={{ width: STICKY_COL_WIDTH, minWidth: STICKY_COL_WIDTH, position: 'sticky', left: 0, zIndex: 900 }}>
                    <div className="flex-grow-1">
                        <div className="d-flex gap-2 mb-1">
                            <input type="number" className="form-control form-control-sm text-center" value={comp.number} onChange={e=>updateComp(comp.code,{number:parseInt(e.target.value)||0})} style={{width:'45px'}} />
                            <input type="text" className="form-control form-control-sm border-0 bg-transparent fw-bold" value={comp.name} onChange={e=>updateComp(comp.code,{name:e.target.value})} />
                        </div>
                        <div className="d-flex gap-2 align-items-center small">
                            <select className="form-select form-select-sm w-auto py-0" value={comp.type} onChange={e=>updateComp(comp.code,{type:e.target.value})}><option>Théorique</option><option>Pratique</option><option>Stage</option></select>
                            <span className="badge bg-light text-dark border ms-auto">{comp.hours}h</span>
                        </div>
                    </div>
                    <button onClick={()=>onUpdateMatrice(matrice.filter(c=>c.code!==comp.code))} className="btn btn-xs btn-outline-danger"><i className="bi bi-trash"></i></button>
                </div>
                <div className="flex-grow-1 position-relative h-100 py-4">
                   <div className="matrix-pill position-absolute rounded shadow-sm d-flex align-items-center" style={{ left: (comp.startWeek*WEEK_COL_WIDTH)+'px', width: ((comp.endWeek-comp.startWeek+1)*WEEK_COL_WIDTH)+'px', backgroundColor: COLORS[comp.type]||'#666', height: '36px', top: '22px', zIndex: 10 }}>
                      <div className="group-ctrl position-absolute d-flex gap-1" style={{ left: '50%', transform: 'translateX(-50%)', top: '-32px' }}>
                          <button onClick={()=>moveComp(comp.code,-1)} className="btn btn-tool shadow-sm" title="Gauche"><i className="bi bi-chevron-left"></i></button>
                          <button onClick={()=>setEditingCompCode(comp.code)} className="btn-tool-main shadow-sm" title="Editer"><i className="bi bi-gear-fill"></i></button>
                          <button onClick={()=>moveComp(comp.code,1)} className="btn btn-tool shadow-sm" title="Droite"><i className="bi bi-chevron-right"></i></button>
                      </div>
                      <div className="d-flex w-100 h-100">
                        {Array.from({ length: comp.endWeek - comp.startWeek + 1 }).map((_, i) => {
                            const w = comp.startWeek + i;
                            return (
                                <div key={i} className="week-block border-end border-white border-opacity-10 h-100 d-flex align-items-center justify-content-center position-relative" style={{ width: WEEK_COL_WIDTH+'px' }}>
                                    <span className="text-white small fw-bold" style={{zIndex: 2}}>{comp.weeklyHours[w] || 0}h</span>
                                    
                                    {/* BOUTON CISEAUX */}
                                    {w < comp.endWeek && (
                                        <button 
                                            className="btn-split position-absolute d-flex align-items-center justify-content-center shadow-sm"
                                            onClick={(e) => { e.stopPropagation(); splitComp(comp.code, w); }}
                                            title="Scinder ici"
                                            style={{ 
                                                right: '-11px', 
                                                top: '50%', 
                                                transform: 'translateY(-50%)', 
                                                width: '22px', 
                                                height: '22px', 
                                                borderRadius: '50%', 
                                                zIndex: 30, 
                                                fontSize: '0.7rem', 
                                                border: '2px solid white',
                                                backgroundColor: '#ff4d4d',
                                                color: 'white',
                                                padding: 0
                                            }}
                                        >
                                            <i className="bi bi-scissors"></i>
                                        </button>
                                    )}
                                </div>
                            );
                        })}
                      </div>
                      <div className="h-ctrl position-absolute w-100 d-flex justify-content-between" style={{left:0, pointerEvents:'none'}}>
                         <div className="d-flex gap-1" style={{marginLeft:'-38px', pointerEvents:'auto'}}><button onClick={()=>resizeComp(comp.code,'start',-1)} className="btn-resize bg-primary text-white shadow-sm">+</button><button onClick={()=>resizeComp(comp.code,'start',1)} className="btn-resize bg-secondary text-white shadow-sm">-</button></div>
                         <div className="d-flex gap-1" style={{marginRight:'-38px', pointerEvents:'auto'}}><button onClick={()=>resizeComp(comp.code,'end',-1)} className="btn-resize bg-secondary text-white shadow-sm">-</button><button onClick={()=>resizeComp(comp.code,'end',1)} className="btn-resize bg-primary text-white shadow-sm">+</button></div>
                      </div>
                   </div>
                </div>
            </div>
          ))}
        </div>
      </div>

      {/* MODALE D'ÉDITION DES DÉTAILS (ROUE DENTÉE) */}
      {editingComp && (
          <div className="modal show d-block" tabIndex={-1} style={{ backgroundColor: 'rgba(0,0,0,0.5)', zIndex: 99999 }}>
              <div className="modal-dialog modal-dialog-centered">
                  <div className="modal-content border-0 shadow-lg">
                      <div className="modal-header bg-dark text-white">
                          <h5 className="modal-title fw-bold">Détails de compétence</h5>
                          <button type="button" className="btn-close btn-close-white" onClick={() => setEditingCompCode(null)}></button>
                      </div>
                      <div className="modal-body p-4">
                          <div className="vstack gap-3">
                              <div className="row g-2">
                                  <div className="col-4"><label className="small fw-bold">Code</label><input type="text" className="form-control" value={editingComp.code} onChange={e => updateComp(editingComp.code, { code: e.target.value })} /></div>
                                  <div className="col-8"><label className="small fw-bold">Nom complet</label><input type="text" className="form-control" value={editingComp.name} onChange={e => updateComp(editingComp.code, { name: e.target.value })} /></div>
                              </div>
                              <div className="row g-2">
                                  <div className="col-6"><label className="small fw-bold">Type</label><select className="form-select" value={editingComp.type} onChange={e => updateComp(editingComp.code, { type: e.target.value })}><option>Théorique</option><option>Pratique</option><option>Stage</option></select></div>
                                  <div className="col-6"><label className="small fw-bold">Local</label><input type="text" className="form-control" value={editingComp.local} onChange={e => updateComp(editingComp.code, { local: e.target.value })} /></div>
                              </div>
                          </div>
                      </div>
                      <div className="modal-footer"><button type="button" className="btn btn-dark fw-bold px-4" onClick={() => setEditingCompCode(null)}>Terminer</button></div>
                  </div>
              </div>
          </div>
      )}

      <style>{`
        .matrix-pill .h-ctrl, .matrix-pill .group-ctrl, .week-block .btn-split { 
            opacity: 0; 
            transition: all 0.2s ease-in-out; 
            pointer-events: none; 
        }
        .matrix-pill:hover .h-ctrl, .matrix-pill:hover .group-ctrl { 
            opacity: 1; 
            pointer-events: auto; 
        }
        .week-block:hover .btn-split { 
            opacity: 1 !important; 
            pointer-events: auto !important; 
            transform: translateY(-50%) scale(1.1) !important;
        }
        .btn-tool-main { 
            width: 32px; 
            height: 32px; 
            background: #212529; 
            color: white; 
            border: 2px solid white; 
            border-radius: 50%; 
            display: flex; 
            align-items: center; 
            justify-content: center; 
            cursor: pointer; 
            box-shadow: 0 4px 8px rgba(0,0,0,0.2);
        }
        .btn-tool { 
            width: 26px; 
            height: 26px; 
            background: white; 
            border: 1px solid #ddd; 
            border-radius: 4px; 
            display: flex; 
            align-items: center; 
            justify-content: center; 
            font-size: 0.75rem; 
        }
        .btn-resize { 
            width: 24px; 
            height: 24px; 
            border-radius: 50%; 
            border: 2px solid white; 
            display: flex; 
            align-items: center; 
            justify-content: center; 
            font-weight: bold; 
            font-size: 0.85rem; 
        }
      `}</style>
    </div>
  );
};
window.MatrixEditor = MatrixEditor;
