
const InternshipPlanning = ({ students, calendar, matrice, isDynamic }) => {
  const { useMemo, useState, useRef, useEffect } = React;
  const { calculateDynamicPlanning } = window;

  const [searchTerm, setSearchTerm] = useState('');
  const scrollContainerRef = useRef(null);

  // Refs pour le drag
  const isDragging = useRef(false);
  const startX = useRef(0);
  const scrollLeftPos = useRef(0);

  const handleMouseDown = (e) => {
    if (!scrollContainerRef.current) return;
    isDragging.current = true;
    scrollContainerRef.current.classList.add('cursor-grabbing');
    scrollContainerRef.current.classList.remove('cursor-grab');
    startX.current = e.pageX;
    scrollLeftPos.current = scrollContainerRef.current.scrollLeft;
  };

  const handleMouseUpOrLeave = () => {
    isDragging.current = false;
    if (scrollContainerRef.current) {
        scrollContainerRef.current.classList.remove('cursor-grabbing');
        scrollContainerRef.current.classList.add('cursor-grab');
    }
  };

  const handleMouseMove = (e) => {
    if (!isDragging.current || !scrollContainerRef.current) return;
    e.preventDefault();
    const x = e.pageX;
    const walk = (x - startX.current) * 2;
    scrollContainerRef.current.scrollLeft = scrollLeftPos.current - walk;
  };

  // --- LOGIC: TIME SCALE ---
  const totalWorkingDays = calendar.length;

  const monthSections = useMemo(() => {
      if (totalWorkingDays === 0) return [];
      
      const sections = [];
      let currentMonth = -1;
      let startIndex = 0;

      calendar.forEach((day, index) => {
          const m = day.date.month();
          if (m !== currentMonth) {
              if (currentMonth !== -1) {
                  sections.push({
                      label: calendar[startIndex].date.format('MMM YY'),
                      left: (startIndex / totalWorkingDays) * 100,
                      width: ((index - startIndex) / totalWorkingDays) * 100
                  });
              }
              currentMonth = m;
              startIndex = index;
          }
      });

      if (startIndex < totalWorkingDays) {
          sections.push({
              label: calendar[startIndex].date.format('MMM YY'),
              left: (startIndex / totalWorkingDays) * 100,
              width: ((totalWorkingDays - startIndex) / totalWorkingDays) * 100
          });
      }

      return sections;
  }, [calendar, totalWorkingDays]);

  const getDayIndex = (date) => {
      const idx = calendar.findIndex(c => c.date.isSame(date, 'day'));
      if (idx !== -1) return idx;
      return -1;
  };

  const getPosition = (date) => {
      if (totalWorkingDays === 0) return 0;
      let idx = getDayIndex(date);
      
      if (idx === -1) {
          if (date.isBefore(calendar[0].date)) return 0;
          if (date.isAfter(calendar[totalWorkingDays - 1].date)) return 100;
          
          let curr = date;
          while(idx === -1 && curr.isAfter(calendar[0].date)) {
              curr = curr.subtract(1, 'day');
              idx = calendar.findIndex(c => c.date.isSame(curr, 'day'));
          }
      }
      return (idx / totalWorkingDays) * 100;
  };

  const getWidth = (start, end) => {
      if (totalWorkingDays === 0) return 0;
      const startIdx = getDayIndex(start);
      let endIdx = getDayIndex(end);
      
      let effectiveStartIdx = startIdx;
      if (effectiveStartIdx === -1) {
           let curr = start;
           while(effectiveStartIdx === -1 && curr.isBefore(end)) {
               curr = curr.add(1, 'day');
               effectiveStartIdx = calendar.findIndex(c => c.date.isSame(curr, 'day'));
           }
           if (effectiveStartIdx === -1) return 0; 
      }

      if (endIdx === -1) {
           let curr = end;
           while(endIdx === -1 && curr.isAfter(start)) {
               curr = curr.subtract(1, 'day');
               endIdx = calendar.findIndex(c => c.date.isSame(curr, 'day'));
           }
      }

      const diff = Math.max(0, endIdx - effectiveStartIdx + 1);
      return (diff / totalWorkingDays) * 100;
  };

  const todayPosition = useMemo(() => {
      if (totalWorkingDays === 0) return -1;
      const today = dayjs();
      if (today.isBefore(calendar[0].date.startOf('day')) || today.isAfter(calendar[totalWorkingDays - 1].date.endOf('day'))) return -1;

      const currentMonth = today.month();
      const currentYear = today.year();
      const startIndex = calendar.findIndex(c => c.date.month() === currentMonth && c.date.year() === currentYear);
      
      if (startIndex === -1) return getPosition(today);

      let nextMonthIndex = totalWorkingDays;
      for (let i = startIndex; i < totalWorkingDays; i++) {
          if (calendar[i].date.month() !== currentMonth || calendar[i].date.year() !== currentYear) {
              nextMonthIndex = i;
              break;
          }
      }
      
      const monthBlockWidth = nextMonthIndex - startIndex;
      const daysInMonth = today.daysInMonth();
      const currentDay = today.date();
      const ratio = (currentDay - 1) / Math.max(1, daysInMonth - 1);
      const interpolatedIndex = startIndex + (ratio * monthBlockWidth);

      return (interpolatedIndex / totalWorkingDays) * 100;
  }, [calendar, totalWorkingDays]);

  const matrixStageComp = useMemo(() => matrice.find(c => c.type === 'Stage'), [matrice]);

  const stageData = useMemo(() => {
    if (calendar.length === 0) return [];

    return students.flatMap(student => {
      const studentFullMatrice = [...matrice, ...(student.customCompetencies || [])];
      const studentStageComps = studentFullMatrice.filter(c => c.type === 'Stage');

      if (studentStageComps.length === 0) return [];

      return studentStageComps.map(stageComp => {
          let start = null;
          let end = null;

          if (isDynamic) {
            const plan = calculateDynamicPlanning(student, calendar, studentFullMatrice);
            const skill = plan.skillStats.find(s => s.code === stageComp.code);
            if (skill && skill.dynamicStart && skill.dynamicEnd) {
                start = skill.dynamicStart;
                end = skill.dynamicEnd;
            }
          } else {
            const studentStartDate = dayjs(student.startDate).startOf('day');
            const startIndex = calendar.findIndex(c => c.date.isSame(studentStartDate, 'day') || c.date.isAfter(studentStartDate));

            if (startIndex !== -1) {
                const stageStartIdx = startIndex + (stageComp.startWeek * 5);
                const stageEndIdx = startIndex + (stageComp.endWeek * 5) + 4;
                if (calendar[stageStartIdx] && calendar[stageEndIdx]) {
                    start = calendar[stageStartIdx].date;
                    end = calendar[stageEndIdx].date;
                }
            }
          }

          if (!start || !end) return null;

          return {
            student,
            stageComp,
            start,
            end
          };
      });
    })
    .filter(item => item !== null)
    .sort((a, b) => a.start.valueOf() - b.start.valueOf());
  }, [students, calendar, matrice, isDynamic]);

  const filteredData = stageData.filter(d => 
    d.student.name.toLowerCase().includes(searchTerm.toLowerCase())
  );

  return (
    <div className="d-flex flex-column h-100 bg-white rounded shadow-sm overflow-hidden">
      <div className="p-3 border-bottom d-flex justify-content-between align-items-end bg-light">
        <div>
          <h2 className="h5 fw-bold text-dark d-flex align-items-center gap-2 mb-1">
            <i className="bi bi-building"></i> Planification des stages
            {matrixStageComp && (
                <span className="badge bg-purple text-white">
                  Baseline: C-{matrixStageComp.number}
                </span>
            )}
          </h2>
        </div>
        <div>
           <input 
              type="text" 
              placeholder="Rechercher un élève..." 
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="form-control form-control-sm w-auto"
           />
        </div>
      </div>

      <div 
        ref={scrollContainerRef} 
        className="flex-grow-1 overflow-auto position-relative d-flex flex-column cursor-grab"
        onMouseDown={handleMouseDown}
        onMouseLeave={handleMouseUpOrLeave}
        onMouseUp={handleMouseUpOrLeave}
        onMouseMove={handleMouseMove}
      >
        <div className="d-flex bg-light border-bottom sticky-top z-2" style={{ minWidth: '1200px', height: '32px' }}>
          <div className="border-end p-2 fw-bold small text-secondary sticky-left bg-light shadow-sm" 
               style={{ width: '200px', minWidth: '200px', position: 'sticky', left: 0, zIndex: 10 }}>
            Étudiant
          </div>
          <div className="flex-grow-1 position-relative">
            {monthSections.map((section, idx) => (
                <div 
                    key={idx}
                    className="position-absolute top-0 bottom-0 border-start px-1 small fw-bold text-muted text-uppercase text-truncate d-flex align-items-center"
                    style={{ left: `${section.left}%`, width: `${section.width}%`, fontSize: '0.7rem' }}
                >
                  {section.label}
                </div>
            ))}
          </div>
        </div>

        <div style={{ minWidth: '1200px' }} className="position-relative">
          
          <div className="position-absolute top-0 bottom-0 start-0 end-0 pe-none" style={{ marginLeft: '200px' }}>
              {todayPosition >= 0 && (
                <div 
                    className="position-absolute top-0 bottom-0 border-start border-2 border-danger z-2"
                    style={{ left: `${todayPosition}%` }}
                >
                    <div className="position-absolute rounded-circle bg-danger" style={{ top: 0, left: '-4px', width: '8px', height: '8px' }}></div>
                </div>
              )}

              {monthSections.map((section, idx) => (
                  <div key={idx} className="position-absolute top-0 bottom-0 border-start" style={{ left: `${section.left}%`, borderColor: '#f0f0f0' }}></div>
              ))}
          </div>

          <div className="position-relative z-1">
            {filteredData.map((data, idx) => {
              const leftPct = getPosition(data.start);
              const widthPct = getWidth(data.start, data.end);
              const isNow = dayjs().isAfter(data.start) && dayjs().isBefore(data.end);

              return (
                <div key={`${data.student.id}-${idx}`} className="d-flex hover-bg-light border-bottom position-relative" style={{ height: '40px' }}>
                  <div className="p-2 small fw-medium text-dark border-end text-truncate sticky-left bg-white shadow-sm d-flex align-items-center gap-1" 
                       style={{ width: '200px', minWidth: '200px', position: 'sticky', left: 0, zIndex: 5 }}>
                    {data.stageComp.isCustom && <i className="bi bi-person-gear text-warning small" title="Stage personnalisé"></i>}
                    <span className="text-truncate">{data.student.name}</span>
                  </div>
                  <div className="flex-grow-1 position-relative my-auto h-100">
                    <div 
                      className={`position-absolute rounded shadow-sm d-flex align-items-center justify-content-center text-white fw-bold overflow-hidden text-nowrap px-2 cursor-help
                        ${isNow ? 'bg-purple border border-purple' : 'bg-purple bg-opacity-75'}`}
                      style={{ 
                        top: '8px', 
                        bottom: '8px',
                        left: `${leftPct}%`, 
                        width: `${widthPct}%`,
                        fontSize: '0.65rem',
                        backgroundColor: isNow ? '#6f42c1' : '#9f7aea'
                      }}
                      title={`${data.stageComp.name} (C-${data.stageComp.number})\nDu ${data.start.format('DD/MM/YYYY')} au ${data.end.format('DD/MM/YYYY')}`}
                    >
                      <span className="text-white">
                        {data.stageComp.isCustom ? '⭐ ' : ''}{data.start.format('D MMM')} - {data.end.format('D MMM')}
                      </span>
                    </div>
                  </div>
                </div>
              );
            })}
            
            {filteredData.length === 0 && (
                <div className="p-5 text-center text-muted fst-italic">Aucun stage planifié trouvé.</div>
            )}
          </div>
        </div>
      </div>
      
      <div className="p-2 bg-light border-top small text-muted d-flex gap-3 justify-content-end">
         <div className="d-flex align-items-center gap-1">
            <div className="rounded" style={{ width: '12px', height: '12px', backgroundColor: '#6f42c1' }}></div> En cours
         </div>
         <div className="d-flex align-items-center gap-1">
            <div className="rounded" style={{ width: '12px', height: '12px', backgroundColor: '#9f7aea' }}></div> Planifié
         </div>
         <div className="d-flex align-items-center gap-1">
            <div className="bg-danger" style={{ width: '2px', height: '12px' }}></div> Aujourd'hui
         </div>
         <div className="d-flex align-items-center gap-1">
            <span className="small fw-bold">⭐ Perso.</span>
         </div>
      </div>
    </div>
  );
};

window.InternshipPlanning = InternshipPlanning;
