
const Inscription = ({ students, onAddStudent, onRemoveStudent, onUpdateStudent, calendar, tosEmail, settings }) => {
  const { useState, useRef, useMemo } = React;
  const { CohortTimeline, parseStudentsCsv } = window;

  const defaultBlocks = settings?.defaultDailyHours || 6;
  const getInitSched = () => {
    const s = {}; for(let i=1; i<=5; i++) s[i] = { slots: Array(defaultBlocks).fill(true) };
    return s;
  };

  const [isPanelOpen, setIsPanelOpen] = useState(false);
  const [editingStudentId, setEditingStudentId] = useState(null);
  const [firstName, setFirstName] = useState('');
  const [lastName, setLastName] = useState('');
  const [startDate, setStartDate] = useState('');
  const [studentEmail, setStudentEmail] = useState('');
  const [sendToTOS, setSendToTOS] = useState(false);
  const [sendToStudent, setSendToStudent] = useState(false);
  
  const [scheduleHistory, setScheduleHistory] = useState([]);
  const [selectedConfigId, setSelectedConfigId] = useState(null);
  const [configStartDate, setConfigStartDate] = useState('');
  const [configType, setConfigType] = useState('FULL_TIME');
  const [configSchedule, setConfigSchedule] = useState(getInitSched());

  const fileInputRef = useRef(null);
  const sortedHistory = useMemo(() => [...scheduleHistory].sort((a,b)=>dayjs(a.startDate).valueOf()-dayjs(b.startDate).valueOf()), [scheduleHistory]);

  const loadConfigIntoEditor = (config) => {
      setSelectedConfigId(config.id);
      setConfigStartDate(config.startDate);
      setConfigType(config.trainingType);
      let base = config.schedule || getInitSched();
      const adapted = {};
      [1,2,3,4,5].forEach(d => {
          let sl = base[d]?.slots || Array(defaultBlocks).fill(true);
          if(sl.length < defaultBlocks) sl = [...sl, ...Array(defaultBlocks-sl.length).fill(true)];
          else if(sl.length > defaultBlocks) sl = sl.slice(0, defaultBlocks);
          adapted[d] = { slots: sl };
      });
      setConfigSchedule(adapted);
  };

  const openNewStudentPanel = () => {
      setEditingStudentId(null); setFirstName(''); setLastName(''); setStudentEmail('');
      const today = dayjs().format('YYYY-MM-DD'); setStartDate(today);
      const init = { id: crypto.randomUUID(), startDate: today, trainingType: 'FULL_TIME', schedule: getInitSched() };
      setScheduleHistory([init]); loadConfigIntoEditor(init); setIsPanelOpen(true);
  };

  const openEditStudentPanel = (student) => {
      setEditingStudentId(student.id); setFirstName(student.firstName); setLastName(student.lastName);
      setStartDate(student.startDate); setStudentEmail(student.studentEmail || '');
      let history = student.scheduleHistory || [{ id: crypto.randomUUID(), startDate: student.startDate, trainingType: student.trainingType||'FULL_TIME', schedule: student.schedule||getInitSched() }];
      setScheduleHistory(history);
      const latest = [...history].sort((a,b)=>dayjs(a.startDate).valueOf()-dayjs(b.startDate).valueOf())[history.length-1];
      if(latest) loadConfigIntoEditor(latest);
      setIsPanelOpen(true);
  };

  const handleSaveStudent = (e) => {
    if(e) e.preventDefault();
    if (!firstName || !lastName || !startDate) return;
    let finalH = [...scheduleHistory];
    if (configStartDate && selectedConfigId) {
        finalH = finalH.map(h => h.id === selectedConfigId ? { id:h.id, startDate:configStartDate, trainingType:configType, schedule:configSchedule } : h);
    }
    finalH.sort((a,b)=>dayjs(a.startDate).valueOf()-dayjs(b.startDate).valueOf());
    const latest = finalH[finalH.length-1];
    const data = { id:editingStudentId||crypto.randomUUID(), firstName, lastName, name:firstName+' '+lastName, startDate, studentEmail, scheduleHistory:finalH, trainingType:latest.trainingType, schedule:latest.schedule, evaluations: editingStudentId?students.find(s=>s.id===editingStudentId)?.evaluations : undefined };
    if(editingStudentId) onUpdateStudent(data); else onAddStudent(data);
    setIsPanelOpen(false);
  };

  const toggleSlot = (day, idx) => {
    const ns = {...configSchedule};
    ns[day].slots[idx] = !ns[day].slots[idx];
    setConfigSchedule(ns);
    if(selectedConfigId) setScheduleHistory(prev => prev.map(h=>h.id===selectedConfigId?{...h, schedule:ns}:h));
  };

  const weeklyHours = useMemo(() => {
      let t = 0; [1,2,3,4,5].forEach(d => { configSchedule[d]?.slots.forEach(v => { if(v) t++; }); });
      return t;
  }, [configSchedule]);

  return (
    <div className="d-flex flex-column h-100 position-relative">
      <div className="d-flex justify-content-between align-items-center mb-4">
         <div><h2 className="h4 fw-bold text-dark mb-0">Gestion des inscriptions</h2></div>
         <div className="d-flex gap-2">
            <button onClick={() => fileInputRef.current?.click()} className="btn btn-sm btn-outline-secondary">Importer</button>
            <input type="file" ref={fileInputRef} onChange={e=>{if(e.target.files[0]){const r=new FileReader(); r.onload=ev=>parseStudentsCsv(ev.target.result).forEach(s=>onAddStudent(s)); r.readAsText(e.target.files[0]);}}} accept=".csv" className="d-none" />
            <button onClick={openNewStudentPanel} className="btn btn-sm btn-primary fw-bold shadow-sm">+ Nouvelle</button>
         </div>
      </div>
      <div className="flex-grow-1 bg-white rounded-3 shadow-sm border overflow-hidden"><CohortTimeline students={students} calendar={calendar.map(c=>c.date)} onEdit={openEditStudentPanel} onDelete={onRemoveStudent} /></div>

      {isPanelOpen && (
          <div className="position-fixed top-0 start-0 w-100 h-100 d-flex justify-content-end" style={{ zIndex: 1050 }}>
              <div className="position-absolute top-0 start-0 w-100 h-100 bg-dark bg-opacity-50" onClick={() => setIsPanelOpen(false)}></div>
              <div className="position-relative bg-white shadow-lg h-100 d-flex flex-column" style={{ width: '100%', maxWidth: '750px' }}>
                  <div className="p-4 border-bottom d-flex justify-content-between align-items-center bg-light"><h3 className="h5 fw-bold mb-0">Fiche Étudiant</h3><button onClick={()=>setIsPanelOpen(false)} className="btn-close"></button></div>
                  <form onSubmit={handleSaveStudent} className="flex-grow-1 overflow-auto p-4 d-flex flex-column gap-4">
                      <div className="row g-3">
                          <div className="col-6"><label className="small fw-bold">Prénom</label><input type="text" value={firstName} onChange={e=>setFirstName(e.target.value)} className="form-control form-control-sm" required /></div>
                          <div className="col-6"><label className="small fw-bold">Nom</label><input type="text" value={lastName} onChange={e=>setLastName(e.target.value)} className="form-control form-control-sm" required /></div>
                          <div className="col-6"><label className="small fw-bold">Entrée</label><input type="date" value={startDate} onChange={e=>setStartDate(e.target.value)} className="form-control form-control-sm" required /></div>
                          <div className="col-6"><label className="small fw-bold">Email</label><input type="email" value={studentEmail} onChange={e=>setStudentEmail(e.target.value)} className="form-control form-control-sm" /></div>
                      </div>
                      <div className="vstack gap-3 mt-2">
                          <h4 className="small fw-bold text-muted text-uppercase border-bottom pb-1">Horaires</h4>
                          <div className="row">
                              <div className="col-md-5 vstack gap-2">
                                  {sortedHistory.map((h, i) => (
                                      <div key={h.id} onClick={()=>loadConfigIntoEditor(h)} className={`p-2 rounded border cursor-pointer small ${h.id===selectedConfigId?'bg-info bg-opacity-10 border-info':'bg-white shadow-sm'}`}>
                                          <div className="fw-bold">{i===0?'Début':'Changement'} : {dayjs(h.startDate).format('DD/MM/YY')}</div>
                                          <div>{h.trainingType==='FULL_TIME'?'Plein':'Partiel'}</div>
                                      </div>
                                  ))}
                                  <button type="button" onClick={()=>{const nid=crypto.randomUUID(); const n={id:nid, startDate:dayjs().format('YYYY-MM-DD'), trainingType:'FULL_TIME', schedule:getInitSched()}; setScheduleHistory([...scheduleHistory, n]); loadConfigIntoEditor(n);}} className="btn btn-xs btn-outline-primary">+ Ajouter un changement</button>
                              </div>
                              <div className="col-md-7">
                                  <div className="bg-light p-3 rounded border h-100 shadow-sm small">
                                      <div className="mb-2"><label className="fw-bold">Date d'effet</label><input type="date" value={configStartDate} onChange={e=>setConfigStartDate(e.target.value)} className="form-control form-control-sm" /></div>
                                      <div className="d-flex gap-3 mb-2">
                                          <div className="form-check"><input className="form-check-input" type="radio" checked={configType==='FULL_TIME'} onChange={()=>setConfigType('FULL_TIME')} id="r1" /><label htmlFor="r1">Plein (30h)</label></div>
                                          <div className="form-check"><input className="form-check-input" type="radio" checked={configType==='PART_TIME'} onChange={()=>setConfigType('PART_TIME')} id="r2" /><label htmlFor="r2">Partiel</label></div>
                                      </div>
                                      {configType==='PART_TIME' && (
                                          <div className="bg-white p-2 rounded border">
                                              <div className="text-center fw-bold mb-1 text-primary">{weeklyHours}h / semaine</div>
                                              <div className="table-responsive">
                                                <table className="table table-sm table-bordered mb-0 text-center align-middle" style={{fontSize:'0.65rem'}}>
                                                    <thead><tr><th></th><th>L</th><th>M</th><th>M</th><th>J</th><th>V</th></tr></thead>
                                                    <tbody>
                                                        {Array.from({length:defaultBlocks}).map((_, idx)=>(
                                                            <tr key={idx}>
                                                                <td className="fw-bold bg-light">{settings?.slotLabels?.[idx] || 'B'+(idx+1)}</td>
                                                                {[1,2,3,4,5].map(d=>(<td key={d}><input type="checkbox" checked={configSchedule[d]?.slots[idx]} onChange={()=>toggleSlot(d,idx)} /></td>))}
                                                            </tr>
                                                        ))}
                                                    </tbody>
                                                </table>
                                              </div>
                                          </div>
                                      )}
                                  </div>
                              </div>
                          </div>
                      </div>
                  </form>
                  <div className="p-3 border-top bg-light d-flex justify-content-end gap-2"><button onClick={()=>setIsPanelOpen(false)} className="btn btn-light">Annuler</button><button onClick={handleSaveStudent} className="btn btn-primary px-4 shadow-sm">Enregistrer</button></div>
              </div>
          </div>
      )}
    </div>
  );
};
window.Inscription = Inscription;
