
const GlobalView = ({ students, calendar, matrice, onSelectStudent }) => {
  const { useMemo } = React;
  const { TabType, calculateDynamicPlanning } = window;
  const today = dayjs().startOf('day');

  const studentProgressData = useMemo(() => {
    return students.map(student => {
      const fullMatrice = [...matrice, ...(student.customCompetencies || [])];
      const plan = calculateDynamicPlanning(student, calendar, fullMatrice);
      
      const skillsWithEvents = fullMatrice.filter(comp => {
          const evaluation = student.evaluations?.[comp.code];
          if (evaluation?.result) return false;
          const hasEval = !!evaluation && evaluation.status !== 'upcoming';
          const hasExam = (student.examEvents || []).some(e => e.competencyCode === comp.code);
          const hasIntern = (student.internshipEvents || []).some(e => e.competencyCode === comp.code);
          return hasEval || hasExam || hasIntern;
      });

      const skillsWithProgress = skillsWithEvents.map(comp => {
        const skill = plan.skillStats.find(s => s.code === comp.code) || { ...comp, status: 'upcoming' };
        const evaluation = student.evaluations?.[comp.code];
        const isCompleted = evaluation?.status === 'completed';
        const pills = [];

        const exams = (student.examEvents || []).filter(e => e.competencyCode === comp.code).sort((a,b)=>dayjs(a.date).valueOf()-dayjs(b.date).valueOf());
        const interns = (student.internshipEvents || []).filter(e => e.competencyCode === comp.code).sort((a,b)=>dayjs(a.date).valueOf()-dayjs(b.date).valueOf());

        const examsTooltip = exams.length > 0 
            ? `Épreuve(s) prévue(s) :\n${exams.map(e => `• ${dayjs(e.date).format('D MMM')} : ${e.duration}h ${e.comment ? `(${e.comment})` : ''}`).join('\n')}`
            : '';
        const internsTooltip = interns.length > 0 
            ? `Stage(s) prévu(s) :\n${interns.map(i => `• ${dayjs(i.date).format('D MMM')} : ${i.duration}h ${i.comment ? `(${i.comment})` : ''}`).join('\n')}`
            : '';

        exams.forEach(ex => {
            const d = dayjs(ex.date);
            if (d.isSame(today, 'day')) pills.push({ type: 'EXAM', label: 'ÉPREUVE AUJOURD\'HUI', icon: 'bi-mortarboard-fill', priority: 3, style: { backgroundColor: '#0d6efd', color: '#fff' }, tooltip: examsTooltip });
            else if (d.isAfter(today, 'day')) pills.push({ type: 'EXAM', label: 'ÉPREUVE FIXÉE', date: d, icon: 'bi-mortarboard-fill', priority: 10, style: { backgroundColor: '#cfe2ff', color: '#084298', border: '1px solid #0d6efd' }, tooltip: examsTooltip });
        });

        // Fix: Removed incorrect 'const' prefix which caused redeclaration error for 'interns' variable
        interns.forEach(int => {
            const d = dayjs(int.date);
            if (d.isSame(today, 'day')) pills.push({ type: 'STAGE', label: 'STAGE EN COURS', icon: 'bi-building-fill', priority: 4, style: { backgroundColor: '#6f42c1', color: '#fff' }, tooltip: internsTooltip });
            else if (d.isAfter(today, 'day')) pills.push({ type: 'STAGE', label: 'STAGE FIXÉ', date: d, icon: 'bi-building-fill', priority: 11, style: { backgroundColor: '#f3e8ff', color: '#6b21a8', border: '1px solid #6f42c1' }, tooltip: internsTooltip });
        });

        const uniquePills = [];
        pills.forEach(p => {
            if (!uniquePills.find(up => up.type === p.type && up.label === p.label)) {
                uniquePills.push(p);
            }
        });

        uniquePills.sort((a,b) => a.priority - b.priority);

        let hDone = 0;
        Object.entries(plan.allocationsByDate).forEach(([dStr, allocs]) => {
          if (dayjs(dStr).isSameOrBefore(today)) {
            const a = allocs.find(x => x.code === comp.code);
            if (a) hDone += a.hours;
          }
        });

        return { ...skill, progressPercent: Math.min(100, Math.round((hDone/comp.hours)*100)), pills: uniquePills };
      });

      return { student, skills: skillsWithProgress };
    }).sort((a,b) => a.student.name.localeCompare(b.student.name));
  }, [students, calendar, matrice, today]);

  return (
    <div className="h-100 overflow-auto">
      <div className="d-flex justify-content-between align-items-center mb-4 px-1">
        <div><h2 className="h4 fw-bold mb-0 text-dark">Suivi global de la cohorte</h2><p className="text-muted small mb-0">Modules actifs (Apprentissage en cours ou épreuves à venir).</p></div>
        <span className="badge bg-light text-dark border small fw-bold px-3 py-2 shadow-sm"><i className="bi bi-calendar3 me-2 text-primary"></i> Aujourd'hui : {today.format('D MMMM YYYY')}</span>
      </div>
      <div className="table-responsive">
        <table className="table table-hover align-middle border shadow-sm rounded overflow-hidden">
          <thead className="bg-light text-secondary small text-uppercase"><tr><th className="px-4 py-3" style={{ width: '250px' }}>Étudiant</th><th className="px-4 py-3">État des modules & Événements</th></tr></thead>
          <tbody>
            {studentProgressData.map(({ student, skills }) => (
              <tr key={student.id}>
                <td className="px-4 py-4 align-top"><button onClick={() => onSelectStudent(student.name, TabType.PROGRESSION)} className="btn btn-link p-0 text-decoration-none fw-bold text-dark text-start d-flex flex-column gap-1 w-100 student-link-card"><div className="d-flex align-items-center gap-2 w-100"><i className="bi bi-person-circle fs-5 text-primary"></i><span className="name-text">{student.name}</span><i className="bi bi-chevron-right ms-auto me-2 small chevron-icon"></i></div><div className="mt-1 badge bg-light text-muted border fw-normal" style={{fontSize:'0.6rem'}}>Entrée : {dayjs(student.startDate).format('DD/MM/YY')}</div></button></td>
                <td className="px-4 py-3">
                  {skills.length > 0 ? (
                    <div className="vstack gap-3">
                      {skills.map(skill => {
                          const isCompleted = skill.status === 'completed';
                          const examEvent = student.examEvents?.find(e => e.competencyCode === skill.code);
                          const examDateStr = examEvent ? dayjs(examEvent.date).format('D MMM') : null;
                          return (
                            <div key={skill.code} className="p-0 rounded overflow-hidden border shadow-sm bg-white">
                              <div className="d-flex align-items-stretch">
                                  <div className="p-2 d-flex flex-column justify-content-center border-end bg-light bg-opacity-50" style={{ width: '80px' }}><span className="small fw-black text-center text-muted" style={{fontSize:'0.6rem'}}>CODE</span><span className="fw-bold text-center text-dark">C-{skill.number}</span></div>
                                  <div className="flex-grow-1 p-2 d-flex flex-column justify-content-center">
                                      <div className="d-flex justify-content-between align-items-center mb-1"><span className="small fw-bold text-dark text-truncate" style={{maxWidth: '200px'}}>{skill.name}</span>{!isCompleted && <span className="badge bg-light text-primary border-0 small fw-black">{skill.progressPercent}%</span>}</div>
                                      {!isCompleted && <div className="progress rounded-pill shadow-none border" style={{ height: '4px' }}><div className="progress-bar bg-primary" style={{ width: `${skill.progressPercent}%` }}></div></div>}
                                      <div className="mt-1 d-flex align-items-center gap-2"><span className="text-muted" style={{ fontSize: '0.65rem' }}><i className="bi bi-calendar-range me-1"></i>{isCompleted ? (<>Fin cours : {dayjs(skill.evaluation?.realEndDate).format('D MMM')} {examDateStr && <span className="text-primary fw-bold mx-1">• Épreuve : {examDateStr}</span>}</>) : `Fin cours prèv. : ${skill.dynamicEnd ? skill.dynamicEnd.format('D MMM') : '--'}`}</span></div>
                                  </div>
                                  <div className="p-2 border-start d-flex align-items-center justify-content-center bg-light bg-opacity-25" style={{ width: '180px' }}>
                                      <div className="vstack gap-1 w-100 px-1">
                                          {skill.pills.slice(0, 2).map((p, i) => (
                                              <div key={i} className="vstack gap-0 align-items-center">
                                                  <div className="rounded px-1 py-1 d-flex align-items-center gap-1 shadow-sm w-100 justify-content-center cursor-help" 
                                                       style={{ ...p.style, fontSize: '0.55rem', fontWeight: 800 }}
                                                       data-bs-toggle="tooltip" title={p.tooltip || p.label}>
                                                      <i className={`bi ${p.icon}`}></i> {p.label}
                                                  </div>
                                                  {p.date && <div className="fw-bold" style={{fontSize:'0.65rem', color: p.type === 'STAGE' ? '#6f42c1' : '#0d6efd'}}>{p.date.format('D MMM')}</div>}
                                              </div>
                                          ))}
                                      </div>
                                  </div>
                              </div>
                            </div>
                          );
                      })}
                    </div>
                  ) : <div className="py-2 text-muted small fst-italic text-center bg-light rounded border border-dashed">Toutes les compétences actives ont été évaluées.</div>}
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>
    </div>
  );
};
window.GlobalView = GlobalView;
