const EvalModal = ({ isOpen, onClose, onSave, competency, student, matrice, fixedDate }) => {
  const { useState, useEffect, useMemo } = React;
  const [selectedCode, setSelectedCode] = useState(competency?.code || '');
  const activeComp = useMemo(() => matrice.find(m => m.code === selectedCode) || competency, [selectedCode, matrice, competency]);

  const [resultDate, setResultDate] = useState(dayjs().format('YYYY-MM-DD'));
  const [result, setResult] = useState('success');
  const [comment, setComment] = useState('');
  const [studentEmail, setStudentEmail] = useState(student?.studentEmail || '');

  useEffect(() => {
    if (isOpen) {
       setSelectedCode(competency?.code || '');
       const initialData = student?.evaluations?.[competency?.code];
       setResultDate(initialData?.resultDate || fixedDate || dayjs().format('YYYY-MM-DD'));
       setResult(initialData?.result || 'success');
       setComment(initialData?.comment || '');
       setStudentEmail(student?.studentEmail || '');
    }
  }, [isOpen, competency, student, fixedDate]);

  if (!isOpen || !activeComp) return null;

  return (
    <div className="modal show d-block" style={{ backgroundColor: 'rgba(0,0,0,0.5)', zIndex: 100000 }}>
      <div className="modal-dialog modal-dialog-centered modal-lg shadow-lg">
        <div className="modal-content border-0 overflow-hidden">
          <div className="modal-header bg-dark text-white"><h5 className="modal-title fw-bold">Validation : {student?.name}</h5><button type="button" className="btn-close btn-close-white" onClick={onClose}></button></div>
          <div className="modal-body p-4">
            <form onSubmit={e => { e.preventDefault(); const cur = student?.evaluations?.[activeComp.code] || {}; onSave({ ...cur, code: activeComp.code, status: 'completed', result, resultDate, comment }, studentEmail); onClose(); }}>
                <div className="mb-4">
                    <span className="badge bg-light text-secondary border mb-2">C-{activeComp.number}</span>
                    <h6 className="fw-bold text-dark mb-0">{activeComp.name}</h6>
                </div>
                <div className="mb-4">
                    <label className="form-label small fw-bold text-secondary text-uppercase" style={{fontSize:'0.65rem'}}>Verdict Final</label>
                    <div className="d-flex gap-2">
                        <input type="radio" className="btn-check" name="rv" id="rs" checked={result === 'success'} onChange={() => setResult('success')} />
                        <label className="btn btn-outline-success w-50 fw-bold py-2 shadow-sm d-flex align-items-center justify-content-center gap-2" htmlFor="rs">RÉUSSITE</label>
                        <input type="radio" className="btn-check" name="rv" id="rf" checked={result === 'failure'} onChange={() => setResult('failure')} />
                        <label className="btn btn-outline-danger w-50 fw-bold py-2 shadow-sm d-flex align-items-center justify-content-center gap-2" htmlFor="rf">ÉCHEC</label>
                    </div>
                </div>
                <div className="mb-4">
                    <label className="form-label small fw-bold text-secondary text-uppercase" style={{fontSize:'0.65rem'}}>Date de l'épreuve / validation</label>
                    <input type="date" required value={resultDate} onChange={e => setResultDate(e.target.value)} className="form-control" />
                </div>
                <div className="mb-4">
                    <label className="form-label small fw-bold text-secondary text-uppercase" style={{fontSize:'0.65rem'}}>Observations</label>
                    <textarea value={comment} onChange={e => setComment(e.target.value)} className="form-control" rows={3} placeholder="Notes sur l'évaluation..." />
                </div>
                <div className="modal-footer px-0 pb-0 pt-3 border-top d-flex justify-content-end gap-2">
                    <button type="button" onClick={onClose} className="btn btn-light fw-bold border shadow-none">Annuler</button>
                    <button type="submit" className="btn btn-dark fw-bold px-4 shadow-sm">Valider le résultat</button>
                </div>
            </form>
          </div>
        </div>
      </div>
    </div>
  );
};

const Evaluations = ({ students, calendar, matrice, isDynamic, onUpdateStudent, globalTosEmail }) => {
  const { useState, useMemo } = React;
  const { calculateDynamicPlanning } = window;
  const [modalOpen, setModalOpen] = useState(false);
  const [selectedStudent, setSelectedStudent] = useState(null);
  const [selectedCompetency, setSelectedCompetency] = useState(null);
  const [fixedDate, setFixedDate] = useState(null);

  const upcomingEvals = useMemo(() => {
    const evalsByWeek = {};
    students.forEach(student => {
      const fullMatrice = [...matrice, ...(student.customCompetencies || [])];
      const scheduledCodes = new Set();

      if (isDynamic) {
        (student.examEvents || []).forEach(exam => {
            const comp = fullMatrice.find(c => c.code === exam.competencyCode);
            if (!comp || student.evaluations?.[comp.code]?.result) return;
            
            scheduledCodes.add(comp.code);
            
            const weekStart = dayjs(exam.date).startOf('isoWeek').format('YYYY-MM-DD');
            if (!evalsByWeek[weekStart]) evalsByWeek[weekStart] = [];
            let entry = evalsByWeek[weekStart].find(e => e.skillCode === comp.code && e.isScheduled);
            if (entry) {
                if(!entry.students.find(s=>s.id===student.id)) entry.students.push({...student, exactDate: exam.date});
            } else {
                evalsByWeek[weekStart].push({ skillCode: comp.code, skillName: comp.name, skillNum: comp.number, date: dayjs(exam.date), students: [{...student, exactDate: exam.date}], isScheduled: true });
            }
        });

        const plan = calculateDynamicPlanning(student, calendar, fullMatrice);
        plan.skillStats.forEach(skill => {
            if (scheduledCodes.has(skill.code)) return;
            if (student.evaluations?.[skill.code]?.result) return;

            if (skill.status === 'in_progress') {
                if (!skill.dynamicEnd) return;
                const weekStart = skill.dynamicEnd.startOf('isoWeek').format('YYYY-MM-DD');
                if (!evalsByWeek[weekStart]) evalsByWeek[weekStart] = [];
                let entry = evalsByWeek[weekStart].find(e => e.skillCode === skill.code && !e.isScheduled);
                if (entry) {
                    if(!entry.students.find(s=>s.id===student.id)) entry.students.push(student);
                } else {
                    evalsByWeek[weekStart].push({ skillCode: skill.code, skillName: skill.name, skillNum: skill.number, date: skill.dynamicEnd, students: [student], isScheduled: false });
                }
            }
        });
      }
    });
    return Object.keys(evalsByWeek).sort().map(k => ({ weekStart: k, evaluations: evalsByWeek[k] }));
  }, [students, calendar, matrice, isDynamic]);

  const handleStudentClick = (s, skillCode, date) => {
      setSelectedStudent(s);
      const fullM = [...matrice, ...(s.customCompetencies || [])];
      setSelectedCompetency(fullM.find(c => c.code === skillCode) || null);
      setFixedDate(date);
      setModalOpen(true);
  };

  return (
    <div className="h-100 overflow-auto">
      <div className="mb-4 border-bottom pb-3 px-1">
        <h2 className="h4 fw-bold mb-1">Modules en cours & évaluations</h2>
        <p className="text-muted small mb-0">Les estimations sont masquées dès qu'une épreuve réelle est fixée dans l'horaire pour un module.</p>
      </div>
      <div className="vstack gap-4">
        {upcomingEvals.length === 0 ? <div className="p-5 text-center text-muted border border-dashed rounded bg-light">Aucun module nécessitant une évaluation pour le moment.</div> : upcomingEvals.map(week => (
          <div key={week.weekStart} className="card shadow-sm border-0 overflow-hidden">
            <div className="card-header bg-white border-bottom py-3 small fw-bold text-muted text-uppercase">Semaine du {dayjs(week.weekStart).format('D MMMM YYYY')}</div>
            <ul className="list-group list-group-flush">
              {week.evaluations.map((ev, i) => (
                <li key={i} className="list-group-item p-4 d-flex flex-column flex-md-row gap-4 align-items-md-center">
                  <div className="d-flex align-items-center gap-3 flex-grow-1">
                    <div className="rounded d-flex flex-column align-items-center justify-content-center border shadow-sm flex-shrink-0 bg-white" style={{ width: '52px', height: '52px' }}>
                        <span className="small fw-black text-secondary lh-1" style={{ fontSize: '0.6rem' }}>C</span><span className="fs-4 fw-black text-dark lh-1">{ev.skillNum}</span>
                    </div>
                    <div>
                        <div className="d-flex align-items-center gap-2 mb-1">
                            <h5 className="h6 fw-bold text-dark mb-0">{ev.skillName}</h5>
                            <span className={`badge ${ev.isScheduled ? 'bg-dark' : 'bg-light text-muted border'} rounded-pill`} style={{fontSize:'0.6rem'}}>{ev.isScheduled ? 'ÉPREUVE' : 'ESTIMATION'}</span>
                        </div>
                        <p className="small text-muted fw-bold mb-0">{ev.isScheduled ? ev.date.format('dddd D MMMM') : 'Fin de cours estimée : '+ev.date.format('D MMM YYYY')}</p>
                    </div>
                  </div>
                  <div className="d-flex flex-wrap gap-2">
                    {ev.students.map(s => (<div key={s.id} onClick={() => handleStudentClick(s, ev.skillCode, s.exactDate || null)} className="badge bg-white text-dark border py-2 px-3 shadow-sm cursor-pointer transition-all hover-bg-dark hover-text-white shadow-none"><i className="bi bi-person-fill me-1"></i>{s.name}</div>))}
                  </div>
                </li>
              ))}
            </ul>
          </div>
        ))}
      </div>
      {selectedStudent && (
        <EvalModal 
            isOpen={modalOpen} 
            onClose={() => setModalOpen(false)} 
            onSave={(data, em) => onUpdateStudent({ ...selectedStudent, studentEmail: em || selectedStudent.studentEmail, evaluations: { ...selectedStudent.evaluations, [data.code]: data } })} 
            competency={selectedCompetency} 
            student={selectedStudent} 
            fixedDate={fixedDate} 
            matrice={[...matrice, ...(selectedStudent?.customCompetencies || [])]} 
        />
      )}
    </div>
  );
};
window.Evaluations = Evaluations;