
const CohortTimeline = ({ students, calendar, onEdit, onDelete }) => {
  const { useRef } = React;
  const scrollContainerRef = useRef(null);
  
  // Refs pour le drag
  const isDragging = useRef(false);
  const startX = useRef(0);
  const scrollLeftPos = useRef(0);

  const handleMouseDown = (e) => {
    if (!scrollContainerRef.current) return;
    isDragging.current = true;
    scrollContainerRef.current.classList.add('cursor-grabbing');
    scrollContainerRef.current.classList.remove('cursor-grab');
    startX.current = e.pageX;
    scrollLeftPos.current = scrollContainerRef.current.scrollLeft;
  };

  const handleMouseUpOrLeave = () => {
    isDragging.current = false;
    if (scrollContainerRef.current) {
        scrollContainerRef.current.classList.remove('cursor-grabbing');
        scrollContainerRef.current.classList.add('cursor-grab');
    }
  };

  const handleMouseMove = (e) => {
    if (!isDragging.current || !scrollContainerRef.current) return;
    e.preventDefault();
    const x = e.pageX;
    const walk = (x - startX.current) * 2;
    scrollContainerRef.current.scrollLeft = scrollLeftPos.current - walk;
  };

  if (calendar.length === 0) return (
      <div className="p-5 text-center text-muted fst-italic border border-dashed rounded bg-light">
          Veuillez configurer le calendrier scolaire pour voir la ligne du temps.
      </div>
  );

  const calendarObjs = calendar.map(d => dayjs(d));
  
  const startMonth = calendarObjs[0].startOf('month');
  const endMonth = calendarObjs[calendarObjs.length - 1].startOf('month');
  const months = [];
  let curr = startMonth;
  while(curr.isBefore(endMonth) || curr.isSame(endMonth, 'month')) {
    months.push(curr);
    curr = curr.add(1, 'month');
  }

  const sortedStudents = [...students].sort((a, b) => dayjs(a.startDate).valueOf() - dayjs(b.startDate).valueOf());

  return (
    <div className="h-100 d-flex flex-column bg-white rounded border overflow-hidden">
      <div 
        ref={scrollContainerRef} 
        className="flex-grow-1 overflow-auto cursor-grab"
        onMouseDown={handleMouseDown}
        onMouseLeave={handleMouseUpOrLeave}
        onMouseUp={handleMouseUpOrLeave}
        onMouseMove={handleMouseMove}
      >
        <table className="table table-bordered mb-0" style={{ borderCollapse: 'separate', borderSpacing: 0 }}>
          <thead className="bg-light sticky-top z-2">
            <tr>
              <th className="px-3 py-3 text-start small fw-bold text-secondary text-uppercase border-end sticky-left bg-light shadow-sm" 
                  style={{ 
                    width: '300px', 
                    minWidth: '300px', 
                    position: 'sticky', 
                    left: 0, 
                    zIndex: 10,
                    borderRight: '2px solid #dee2e6'
                  }}>
                  Étudiants & Actions
              </th>
              {months.map(m => (
                <th key={m.format('YYYY-MM')} className="px-2 py-3 text-center small fw-bold text-muted text-uppercase" style={{ minWidth: '80px', fontSize: '0.7rem' }}>
                  {m.format('MMM YY')}
                </th>
              ))}
            </tr>
          </thead>
          <tbody>
            {sortedStudents.map(student => {
              const studentDate = dayjs(student.startDate);
              const today = dayjs();
              
              let currentStatus = 'FULL_TIME';
              if (student.scheduleHistory && student.scheduleHistory.length > 0) {
                  const active = student.scheduleHistory
                     .filter(c => today.isSame(dayjs(c.startDate), 'day') || today.isAfter(dayjs(c.startDate)))
                     .sort((a, b) => dayjs(b.startDate).valueOf() - dayjs(a.startDate).valueOf())[0];
                  
                  if (active) currentStatus = active.trainingType;
                  else if (student.scheduleHistory.length > 0) currentStatus = student.scheduleHistory[0].trainingType;
              }

              return (
                <tr key={student.id} className="hover-bg-light">
                  <td className="px-3 py-2 border-end sticky-left bg-white shadow-sm" 
                      style={{ 
                        position: 'sticky', 
                        left: 0, 
                        zIndex: 5,
                        borderRight: '2px solid #dee2e6'
                      }}>
                    <div className="d-flex justify-content-between align-items-center">
                        <div className="d-flex flex-column cursor-pointer" onClick={() => onEdit(student)}>
                            <div className="fw-bold text-dark small text-primary-hover">{student.name}</div>
                            <div className="d-flex align-items-center gap-2 mt-1">
                                {currentStatus === 'PART_TIME' ? (
                                    <span className="badge bg-purple text-white border border-purple text-uppercase shadow-sm" style={{ fontSize: '0.65rem' }}>Partiel</span>
                                ) : (
                                    <span className="badge bg-primary text-white border border-primary text-uppercase shadow-sm" style={{ fontSize: '0.65rem' }}>Plein</span>
                                )}
                                <span className="text-muted" style={{ fontSize: '0.65rem' }}>
                                    Entrée : {studentDate.format('DD/MM/YYYY')}
                                </span>
                            </div>
                        </div>
                        
                        <div className="d-flex gap-1 opacity-50 hover-opacity-100">
                             <button 
                                onClick={() => onEdit(student)}
                                className="btn btn-sm btn-outline-primary py-0 px-1 border-0"
                                title="Modifier la fiche"
                             >
                                 <i className="bi bi-pencil"></i>
                             </button>
                             <button 
                                onClick={(e) => { e.stopPropagation(); if(window.confirm(`Supprimer ${student.name} ?`)) onDelete(student.id); }}
                                className="btn btn-sm btn-outline-danger py-0 px-1 border-0"
                                title="Supprimer"
                             >
                                 <i className="bi bi-trash"></i>
                             </button>
                        </div>
                    </div>
                  </td>

                  {months.map(m => {
                    const isEntryMonth = studentDate.isSame(m, 'month');
                    return (
                      <td key={m.format('YYYY-MM')} className="px-1 py-2 text-center position-relative align-middle">
                        {isEntryMonth && (
                          <div 
                            className="bg-warning text-dark small fw-black rounded-circle d-flex align-items-center justify-content-center mx-auto shadow-sm"
                            style={{ width: '24px', height: '24px', fontSize: '0.65rem', cursor: 'help' }}
                            title={`Date d'entrée : ${studentDate.format('DD MMMM YYYY')}`}
                          >
                            {studentDate.date()}
                          </div>
                        )}
                      </td>
                    );
                  })}
                </tr>
              );
            })}
            {sortedStudents.length === 0 && (
                <tr>
                    <td colSpan={months.length + 1} className="p-5 text-center text-muted fst-italic">
                        Aucun étudiant inscrit pour le moment.
                    </td>
                </tr>
            )}
          </tbody>
        </table>
      </div>
      <div className="bg-light p-2 small text-muted d-flex justify-content-end gap-3 border-top">
          <div className="d-flex align-items-center gap-1">
              <div className="rounded-circle bg-warning d-flex align-items-center justify-content-center text-dark fw-bold" style={{ width: '16px', height: '16px', fontSize: '0.5rem' }}>J</div>
              <span>Jour d'entrée</span>
          </div>
      </div>
    </div>
  );
};

window.CohortTimeline = CohortTimeline;
