
const CalendarManager = ({ calendar, onUpdateCalendar, settings, onUpdateSettings }) => {
  const { useState, useMemo, useRef } = React;
  const [currentYear, setCurrentYear] = useState(dayjs().year());
  const [selectedMonth, setSelectedMonth] = useState(dayjs().month());
  const fileInputRef = useRef(null);
  const FIXED_HOURS = settings?.defaultDailyHours || 6;

  const monthDates = useMemo(() => {
    const start = dayjs().year(currentYear).month(selectedMonth).startOf('month'), end = start.endOf('month'), days = [];
    const startDay = start.day() === 0 ? 6 : start.day() - 1; 
    for (let i = 0; i < startDay; i++) days.push(start.subtract(startDay - i, 'day'));
    for (let d = 1; d <= end.date(); d++) days.push(start.date(d));
    return days;
  }, [currentYear, selectedMonth]);

  const toggleDay = (date) => {
    const idx = calendar.findIndex(c => c.date.isSame(date, 'day'));
    let newCal = [...calendar];
    if (idx >= 0) newCal.splice(idx, 1);
    else newCal.push({ date: date, hours: FIXED_HOURS });
    onUpdateCalendar(newCal.sort((a, b) => a.date.valueOf() - b.date.valueOf()));
  };

  const handleBulkToggle = () => {
    const daysInMonth = monthDates.filter(d => d.month() === selectedMonth && d.day() !== 0 && d.day() !== 6);
    const activeInMonth = daysInMonth.filter(d => calendar.some(c => c.date.isSame(d, 'day')));
    const turnOff = activeInMonth.length > 0;
    
    let newCal = [...calendar];
    daysInMonth.forEach(d => {
        const idx = newCal.findIndex(c => c.date.isSame(d, 'day'));
        if (turnOff) { if (idx >= 0) newCal.splice(idx, 1); }
        else { if (idx < 0) newCal.push({ date: d, hours: FIXED_HOURS }); }
    });
    onUpdateCalendar(newCal.sort((a, b) => a.date.valueOf() - b.date.valueOf()));
  };

  const updateSlotLabels = (val) => {
    onUpdateSettings({ ...settings, slotLabels: val.split('\n') });
  };

  return (
    <div className="d-flex flex-column h-100 bg-light p-4 overflow-auto">
      <div className="d-flex justify-content-between align-items-start mb-4">
        <div><h2 className="h4 fw-bold text-dark mb-0 text-uppercase" style={{letterSpacing:'0.05em'}}>Configuration du calendrier</h2><p className="text-muted small">Sélectionnez les jours d'ouverture du centre.</p></div>
        <div className="d-flex gap-2">
            <div className="bg-white p-2 border rounded shadow-sm d-flex align-items-center gap-2">
                <label className="small fw-bold text-secondary text-uppercase mb-0">Blocs/Jour :</label>
                <input type="number" className="form-control form-control-sm w-auto text-center fw-bold border-0 bg-light" min="1" max="12" value={FIXED_HOURS} onChange={e=>onUpdateSettings({...settings, defaultDailyHours: parseInt(e.target.value)||6})} />
            </div>
            <button onClick={() => fileInputRef.current?.click()} className="btn btn-sm btn-outline-primary shadow-sm"><i className="bi bi-folder-symlink"></i> Importer CSV</button>
            <input type="file" ref={fileInputRef} onChange={(e) => {
                if (e.target.files?.[0]) {
                    const reader = new FileReader();
                    reader.onload = (ev) => { if (typeof ev.target?.result === 'string') onUpdateCalendar(window.parseCalendarCsv(ev.target.result, FIXED_HOURS)); };
                    reader.readAsText(e.target.files[0]);
                }
            }} accept=".csv" className="d-none" />
        </div>
      </div>

      <div className="bg-white rounded-3 shadow-sm border p-4 mb-4">
          <div className="d-flex align-items-center gap-3 mb-4">
              <div className="btn-group shadow-sm"><button onClick={() => setCurrentYear(currentYear - 1)} className="btn btn-sm btn-light border">◀</button><button className="btn btn-sm btn-white border px-3 fw-bold">{currentYear}</button><button onClick={() => setCurrentYear(currentYear + 1)} className="btn btn-sm btn-light border">▶</button></div>
              <div className="d-flex flex-grow-1 overflow-auto gap-1 pb-2">{Array.from({length: 12}).map((_, i) => (<button key={i} onClick={() => setSelectedMonth(i)} className={`btn btn-sm fw-bold ${selectedMonth === i ? 'btn-primary shadow-sm' : 'btn-light text-secondary border'}`}>{dayjs().month(i).format('MMM')}</button>))}</div>
          </div>

          <div className="d-grid gap-1 bg-light border rounded overflow-hidden" style={{ gridTemplateColumns: 'repeat(5, 1fr)' }}>
                {['Lun', 'Mar', 'Mer', 'Jeu', 'Ven'].map(d => <div key={d} className="bg-white p-2 text-center small fw-bold text-uppercase text-secondary border-bottom">{d}</div>)}
                {monthDates.map((date, i) => {
                    if (date.day() === 0 || date.day() === 6) return null;
                    const isCur = date.month() === selectedMonth;
                    const day = calendar.find(c => c.date.isSame(date, 'day'));
                    return (
                        <div key={i} onClick={() => isCur && toggleDay(date)} className={`p-2 cursor-pointer border-end border-bottom ${!isCur ? 'bg-light opacity-25' : 'bg-white'} ${day ? 'bg-primary bg-opacity-10 border-primary' : 'hover-bg-light'}`} style={{ minHeight: '100px' }}>
                            {isCur && (
                                <>
                                    <div className={`text-end small fw-bold mb-2 ${day ? 'text-primary' : 'text-muted'}`}>{date.date()}</div>
                                    {day ? (
                                        <div className="vstack gap-1 align-items-center">
                                            <span className="badge bg-primary shadow-sm mb-1">{day.hours}h</span>
                                            <div className="d-flex flex-wrap gap-1 justify-content-center px-2">
                                                {Array.from({length: day.hours}).map((_, idx) => <div key={idx} className="bg-primary rounded-circle" style={{width: 5, height: 5}}></div>)}
                                            </div>
                                        </div>
                                    ) : <div className="text-center opacity-25 mt-2 fw-black small text-muted" style={{fontSize:'0.6rem'}}>OFF</div>}
                                </>
                            )}
                        </div>
                    );
                })}
          </div>
          <div className="mt-4 d-flex justify-content-between align-items-center border-top pt-3">
              <span className="text-muted small">Sélectionnés ce mois : <strong>{monthDates.filter(d => d.month() === selectedMonth && calendar.some(c => c.date.isSame(d, 'day'))).length}</strong> j</span>
              <button onClick={handleBulkToggle} className="btn btn-primary btn-sm fw-bold px-4 shadow-sm rounded-pill">Activer/Désactiver le mois</button>
          </div>
      </div>

      <div className="card shadow-sm border-0 mt-2">
          <div className="card-header bg-dark text-white fw-bold small text-uppercase py-2">Libellés des plages horaires (Optionnel)</div>
          <div className="card-body">
              <p className="text-muted small mb-3">Saisissez les libellés pour chaque bloc (un par ligne). Ces libellés s'afficheront dans l'onglet Inscriptions.</p>
              <textarea 
                className="form-control" 
                rows={FIXED_HOURS} 
                placeholder={`Exemple :\n08:15 - 09:15\n09:15 - 10:15...`}
                value={settings?.slotLabels?.join('\n') || ''}
                onChange={e => updateSlotLabels(e.target.value)}
              />
          </div>
      </div>
    </div>
  );
};
window.CalendarManager = CalendarManager;
