
const App = () => {
  const { useState, useMemo, useEffect } = React;
  const { TabType, AppMode } = window;
  const { generateProjectFile, parseProjectFile } = window;
  
  const { 
      Navbar, Sidebar, Header, Inscription, CalendarManager, MatrixViewer, MatrixEditor,
      SkillProgression, MonthlySchedule, InternshipPlanning, RoomOccupancy, Evaluations, GlobalView 
  } = window;

  const [activeTab, setActiveTab] = useState(TabType.VUE_GLOBALE);
  const [isDynamic, setIsDynamic] = useState(true);
  const [tosEmail, setTosEmail] = useState('');
  
  const [calendar, setCalendar] = useState([]);
  const [matrice, setMatrice] = useState([]);
  const [students, setStudents] = useState([]);
  const [projectName, setProjectName] = useState("Macroplanification FP");
  const [selectedStudentName, setSelectedStudentName] = useState('');
  
  const [settings, setSettings] = useState({
      defaultDailyHours: 6,
      isDynamic: true,
      slotLabels: ["08:15-09:15", "09:15-10:15", "10:30-11:30", "12:30-13:30", "13:30-14:30", "14:45-15:45"]
  });

  const [showSaveWarning, setShowSaveWarning] = useState(false);
  const [dontShowSaveWarningAgain, setDontShowSaveWarningAgain] = useState(false);

  // GESTION GLOBALE DES TOOLTIPS (Solution au bug "closest" et rafraîchissement)
  useEffect(() => {
    const initTooltips = () => {
        if (!window.bootstrap) return;
        
        // Nettoyage rigoureux
        const tooltips = document.querySelectorAll('.tooltip');
        tooltips.forEach(t => t.remove());

        const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
        tooltipTriggerList.forEach(el => {
            try {
                // Utilisation de container: 'body' pour éviter les conflits de DOM Moodle
                new window.bootstrap.Tooltip(el, { 
                    container: 'body',
                    trigger: 'hover',
                    boundary: 'viewport'
                });
            } catch (err) {
                console.warn("Erreur Tooltip sur :", el, err);
            }
        });
    };

    const timer = setTimeout(initTooltips, 500); 
    return () => clearTimeout(timer);
  }, [activeTab, students, matrice, calendar, selectedStudentName]);

  const selectedStudent = useMemo(() => students.find(s => s.name === selectedStudentName), [students, selectedStudentName]);

  const executeSave = async () => {
    const finalSettings = { ...settings, isDynamic, tosEmail };
    const json = generateProjectFile(students, calendar, matrice, settings.defaultDailyHours, projectName, isDynamic, finalSettings);
    const safeName = `${projectName.replace(/[^a-z0-9]/gi, '_').toLowerCase()}.json`;

    // TENTATIVE DE SAUVEGARDE AVEC CHOIX DE DOSSIER (API Native)
    if (typeof window.showSaveFilePicker === 'function') {
      try {
        const handle = await window.showSaveFilePicker({
          suggestedName: safeName,
          types: [{
            description: 'Fichier JSON de Planification',
            accept: { 'application/json': ['.json'] },
          }],
        });
        const writable = await handle.createWritable();
        await writable.write(json);
        await writable.close();
        return; // Succès
      } catch (e) {
        if (e.name === 'AbortError') return;
        console.warn("API Native non disponible ou refusée, passage au téléchargement standard.");
      }
    }

    // FALLBACK : Téléchargement classique (Dossier Téléchargements par défaut)
    const blob = new Blob([json], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = safeName;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
  };

  const handleSaveClick = () => {
      const skipWarning = localStorage.getItem('fp_skip_save_warning') === 'true';
      if (skipWarning) executeSave();
      else { setDontShowSaveWarningAgain(false); setShowSaveWarning(true); }
  };

  const handleLoadProject = async (file) => {
    const text = await file.text();
    const data = parseProjectFile(text);
    setStudents(data.students);
    setCalendar(data.calendar);
    setMatrice(data.matrix);
    if (data.projectName) setProjectName(data.projectName);
    if (data.settings) {
        setSettings(data.settings);
        setIsDynamic(data.settings.isDynamic ?? true);
        setTosEmail(data.settings.tosEmail || '');
    }
    setActiveTab(TabType.VUE_GLOBALE);
  };

  return (
    <div className="d-flex flex-column vh-100 overflow-hidden bg-light">
      <Navbar projectName={projectName} setProjectName={setProjectName} tosEmail={tosEmail} setTosEmail={setTosEmail} onNew={() => { if(confirm("Nouveau projet ?")) { setStudents([]); setCalendar([]); setMatrice([]); } }} onSave={handleSaveClick} onLoad={handleLoadProject} isDynamic={isDynamic} setIsDynamic={setIsDynamic} />
      <div className="d-flex flex-grow-1 overflow-hidden">
        <Sidebar activeTab={activeTab} setActiveTab={setActiveTab} />
        <div className="flex-grow-1 d-flex flex-column" style={{ minWidth: 0 }}>
          <Header mode={AppMode.COHORT} students={students} selectedStudentName={selectedStudentName} setSelectedStudentName={setSelectedStudentName} activeTab={activeTab} />
          <main className="flex-grow-1 overflow-y-auto p-4">
            <div className="bg-white rounded-3 shadow-sm p-4 h-100 d-flex flex-column">
              {activeTab === TabType.VUE_GLOBALE && <GlobalView students={students} calendar={calendar} matrice={matrice} onSelectStudent={(n, t) => { setSelectedStudentName(n); setActiveTab(t); }} />}
              {activeTab === TabType.INSCRIPTION && <Inscription students={students} calendar={calendar} onAddStudent={s => setStudents([...students, s])} onRemoveStudent={id => setStudents(students.filter(s => s.id !== id))} onUpdateStudent={s => setStudents(students.map(st => st.id === s.id ? s : st))} tosEmail={tosEmail} settings={settings} />}
              {activeTab === TabType.CALENDRIER && <CalendarManager calendar={calendar} onUpdateCalendar={setCalendar} settings={settings} onUpdateSettings={setSettings} />}
              {activeTab === TabType.TIMELINE && <MatrixViewer matrice={matrice} onUpdateMatrice={setMatrice} student={selectedStudent} calendar={calendar} isDynamic={isDynamic} />}
              {activeTab === TabType.MATRICE && <MatrixEditor matrice={matrice} onUpdateMatrice={setMatrice} />}
              {activeTab === TabType.PROGRESSION && <SkillProgression student={selectedStudent} calendar={calendar} matrice={matrice} tosEmail={tosEmail} onUpdateStudent={s => setStudents(students.map(st => st.id === s.id ? s : st))} onUpdateMatrice={setMatrice} isDynamic={isDynamic} />}
              {activeTab === TabType.HORAIRE && <MonthlySchedule student={selectedStudent} calendar={calendar} matrice={matrice} isDynamic={isDynamic} onUpdateStudent={s => setStudents(students.map(st => st.id === s.id ? s : st))} />}
              {activeTab === TabType.STAGE && <InternshipPlanning students={students} calendar={calendar} matrice={matrice} isDynamic={isDynamic} />}
              {activeTab === TabType.LOCAUX && <RoomOccupancy students={students} calendar={calendar} matrice={matrice} isDynamic={isDynamic} />}
              {activeTab === TabType.EVALUATIONS && <Evaluations students={students} calendar={calendar} matrice={matrice} isDynamic={isDynamic} globalTosEmail={tosEmail} onUpdateStudent={s => setStudents(students.map(st => st.id === s.id ? s : st))} />}
            </div>
          </main>
        </div>
      </div>
      {showSaveWarning && (
        <div className="modal show d-block" style={{ zIndex: 99999 }}>
          <div className="modal-dialog modal-dialog-centered">
            <div className="modal-content border-0 shadow-lg">
              <div className="modal-header bg-warning bg-opacity-10 border-bottom-0"><h5 className="modal-title h6 fw-bold text-dark d-flex align-items-center gap-2"><i className="bi bi-shield-lock-fill text-warning"></i> Confidentialité</h5><button type="button" className="btn-close" onClick={() => setShowSaveWarning(false)}></button></div>
              <div className="modal-body"><p className="small text-muted mb-3">Ce document contient des <strong>données nominatives</strong>.</p><div className="alert alert-light border small text-secondary"><ul className="mb-0 ps-3"><li>Conservez ce fichier dans un espace sécurisé.</li><li>Respectez la politique de gestion documentaire de votre CSS.</li></ul></div><div className="form-check mt-3"><input className="form-check-input" type="checkbox" id="da" checked={dontShowSaveWarningAgain} onChange={(e) => setDontShowSaveWarningAgain(e.target.checked)} /><label className="form-check-label small text-muted" htmlFor="da">Ne plus afficher</label></div></div>
              <div className="modal-footer border-top-0 pt-0"><button type="button" className="btn btn-light btn-sm fw-bold text-secondary" onClick={() => setShowSaveWarning(false)}>Annuler</button><button type="button" className="btn btn-primary btn-sm fw-bold" onClick={() => { if(dontShowSaveWarningAgain) localStorage.setItem('fp_skip_save_warning', 'true'); setShowSaveWarning(false); executeSave(); }}>Accepter</button></div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};
window.App = App;
