
import React, { useState, useMemo } from 'react';
import { AppMode, TabType } from './types.js';
import { generateProjectFile, parseProjectFile } from './utils/dataParser.js';

import Navbar from './components/Navbar.js'; 
import Sidebar from './components/Sidebar.js';
import Header from './components/Header.js';
import Inscription from './components/Inscription.js';
import CalendarManager from './components/CalendarManager.js';
import MatrixViewer from './components/MatrixViewer.js';
import SkillProgression from './components/SkillProgression.js';
import MonthlySchedule from './components/MonthlySchedule.js';
import InternshipPlanning from './components/InternshipPlanning.js';
import RoomOccupancy from './components/RoomOccupancy.js';
import Evaluations from './components/Evaluations.js';

const App = () => {
  const [activeTab, setActiveTab] = useState(TabType.INSCRIPTION);
  const [isDynamic, setIsDynamic] = useState(false);
  const [calendar, setCalendar] = useState([]);
  const [matrice, setMatrice] = useState([]);
  const [students, setStudents] = useState([]);
  const [defaultDailyHours, setDefaultDailyHours] = useState(6);
  const [projectName, setProjectName] = useState("Macroplanification FP");
  const [selectedStudentName, setSelectedStudentName] = useState('');
  
  const selectedStudent = useMemo(() => {
    return students.find(s => s.name === selectedStudentName);
  }, [students, selectedStudentName]);

  const handleNewProject = () => {
    setStudents([]);
    setCalendar([]);
    setMatrice([]);
    setProjectName("Macroplanification FP");
    setIsDynamic(false);
    setActiveTab(TabType.INSCRIPTION);
  };

  const handleSaveProject = async () => {
    const json = generateProjectFile(students, calendar, matrice, defaultDailyHours, projectName, isDynamic);
    try {
      const blob = new Blob([json], { type: 'application/json' });
      const url = URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      const safeName = projectName.replace(/[^a-z0-9]/gi, '_').toLowerCase();
      a.download = `${safeName || 'planification'}.json`;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      URL.revokeObjectURL(url);
    } catch (e) {
      console.error("Erreur lors du téléchargement du fichier", e);
    }
  };

  const handleLoadProject = async (file) => {
    try {
      const text = await file.text();
      const data = parseProjectFile(text);
      setStudents(data.students);
      setCalendar(data.calendar);
      setMatrice(data.matrix);
      setDefaultDailyHours(data.defaultDailyHours);
      if (data.projectName) setProjectName(data.projectName);
      setIsDynamic(data.isDynamic);
      setActiveTab(TabType.INSCRIPTION);
    } catch (e) {
      console.error(e);
    }
  };

  const handleAddStudent = (student) => {
    setStudents(prev => [...prev, student]);
  };

  const handleRemoveStudent = (id) => {
    setStudents(prev => prev.filter(s => s.id !== id));
    if (selectedStudent?.id === id) setSelectedStudentName('');
  };

  const handleUpdateStudent = (updatedStudent) => {
    setStudents(prev => prev.map(s => s.id === updatedStudent.id ? updatedStudent : s));
  };

  const calendarDatesOnly = useMemo(() => calendar.map(c => c.date), [calendar]);

  return (
    <div className="d-flex flex-column vh-100 overflow-hidden bg-light">
      <Navbar 
        projectName={projectName}
        setProjectName={setProjectName}
        onNew={handleNewProject}
        onSave={handleSaveProject}
        onLoad={handleLoadProject}
        isDynamic={isDynamic}
        setIsDynamic={setIsDynamic}
      />
      <div className="d-flex flex-grow-1 overflow-hidden">
        <Sidebar 
          activeTab={activeTab} 
          setActiveTab={setActiveTab}
        />
        <div className="flex-grow-1 d-flex flex-column" style={{ minWidth: 0 }}>
          <Header 
            mode={AppMode.COHORT}
            students={students}
            selectedStudentName={selectedStudentName}
            setSelectedStudentName={setSelectedStudentName}
            calendar={calendarDatesOnly}
            activeTab={activeTab}
          />
          <main className="flex-grow-1 overflow-y-auto p-4">
            <div className="bg-white rounded-3 shadow-sm p-4 h-100 d-flex flex-column">
              {activeTab === TabType.INSCRIPTION && (
                <Inscription 
                  students={students}
                  calendar={calendar}
                  onAddStudent={handleAddStudent}
                  onRemoveStudent={handleRemoveStudent}
                  onUpdateStudent={handleUpdateStudent}
                />
              )}
              {activeTab === TabType.CALENDRIER && (
                <CalendarManager 
                  calendar={calendar}
                  onUpdateCalendar={setCalendar}
                  defaultDailyHours={defaultDailyHours}
                />
              )}
              {activeTab === TabType.MATRICE && (
                <MatrixViewer 
                  matrice={matrice}
                  onUpdateMatrice={setMatrice}
                  student={selectedStudent}
                  calendar={calendar}
                  isDynamic={isDynamic}
                />
              )}
              {activeTab === TabType.PROGRESSION && (
                <SkillProgression 
                  student={selectedStudent} 
                  calendar={calendar} 
                  matrice={matrice} 
                  onUpdateStudent={handleUpdateStudent}
                  isDynamic={isDynamic}
                />
              )}
              {activeTab === TabType.HORAIRE && (
                <MonthlySchedule 
                  student={selectedStudent} 
                  calendar={calendar} 
                  matrice={matrice}
                  isDynamic={isDynamic} 
                />
              )}
              {activeTab === TabType.STAGE && (
                <InternshipPlanning 
                  students={students} 
                  calendar={calendar} 
                  matrice={matrice} 
                  isDynamic={isDynamic}
                />
              )}
              {activeTab === TabType.LOCAUX && (
                <RoomOccupancy 
                  students={students} 
                  calendar={calendar} 
                  matrice={matrice} 
                  isDynamic={isDynamic}
                />
              )}
              {activeTab === TabType.EVALUATIONS && (
                <Evaluations 
                  students={students} 
                  calendar={calendar} 
                  matrice={matrice} 
                  isDynamic={isDynamic}
                  onUpdateStudent={handleUpdateStudent}
                />
              )}
            </div>
          </main>
        </div>
      </div>
    </div>
  );
};
export default App;
