import React from 'react';
import ReactDOM from 'react-dom/client';

// === From: src/utils.ts ===
const GRID_SIZE = 20;
const snapToGrid = (value) => Math.round(value / GRID_SIZE) * GRID_SIZE;
const generateOrthogonalPathString = (points) => {
    if (!points || points.length === 0) return "";
    let d = `M ${points[0].x} ${points[0].y}`;
    for (let i = 1; i < points.length; i++) {
        d += ` L ${points[i].x} ${points[i].y}`;
    }
    return d;
};

// === From: src/lib/circuitUtils.ts ===
function distSq(p1, p2) {
    return Math.pow(p1.x - p2.x, 2) + Math.pow(p1.y - p2.y, 2);
}
function getClosestPointOnSegment(p, v, w) {
    const l2 = distSq(v, w);
    if (l2 === 0) return v;
    let t = ((p.x - v.x) * (w.x - v.x) + (p.y - v.y) * (w.y - v.y)) / l2;
    t = Math.max(0, Math.min(1, t));
    return { x: v.x + t * (w.x - v.x), y: v.y + t * (w.y - v.y) };
}
function distToSegmentSquared(p, v, w) {
    const closestPoint = getClosestPointOnSegment(p, v, w);
    return distSq(p, closestPoint);
}
const simplifyNodes = (wiresToUpdate, componentsToUpdate, nodesToProcess) => {
    let currentWires = [...wiresToUpdate];
    let currentComponents = [...componentsToUpdate];
    while (nodesToProcess.size > 0) {
        const nodeId = nodesToProcess.values().next().value;
        nodesToProcess.delete(nodeId);
        if (!currentComponents.find(c => c.id === nodeId)) continue;
        const connectedWires = currentWires.filter(w => w.from.componentId === nodeId || w.to.componentId === nodeId);
        if (connectedWires.length === 2) {
            const [wire1, wire2] = connectedWires;
            const endpoint1 = wire1.from.componentId === nodeId ? wire1.to : wire1.from;
            const path1 = wire1.from.componentId === nodeId ? [...wire1.path].reverse() : [...wire1.path];
            const endpoint2 = wire2.from.componentId === nodeId ? wire2.to : wire2.from;
            const path2 = wire2.to.componentId === nodeId ? [...wire2.path].reverse() : [...wire2.path];
            const newPath = [...path1.slice(0, -1), ...path2];
            const newWire = {
                id: `${endpoint1.componentId}:${endpoint1.portIndex}-${endpoint2.componentId}:${endpoint2.portIndex}`,
                from: endpoint1,
                to: endpoint2,
                path: newPath,
            };
            currentWires = currentWires.filter(w => w.id !== wire1.id && w.id !== wire2.id);
            const existingWireIndex = currentWires.findIndex(w => (w.from.componentId === endpoint1.componentId && w.from.portIndex === endpoint1.portIndex && w.to.componentId === endpoint2.componentId && w.to.portIndex === endpoint2.portIndex) || (w.from.componentId === endpoint2.componentId && w.from.portIndex === endpoint2.portIndex && w.to.componentId === endpoint1.componentId && w.to.portIndex === endpoint1.portIndex));
            if (existingWireIndex === -1) {
                currentWires.push(newWire);
            }
            currentComponents = currentComponents.filter(c => c.id !== nodeId);
            const comp1 = currentComponents.find(c => c.id === endpoint1.componentId);
            if (comp1?.type === 'node') nodesToProcess.add(comp1.id);
            const comp2 = currentComponents.find(c => c.id === endpoint2.componentId);
            if (comp2?.type === 'node') nodesToProcess.add(comp2.id);
        }
    }
    return { newWires: currentWires, newComponents: currentComponents };
};

// === From: src/lib/componentUtils.ts ===
const normalizeAndValidateConfig = (config) => {
    if (!config || typeof config !== 'object') return null;
    if (config.svg && typeof config.svg === 'object' && !Array.isArray(config.svg) && config.svg.props && Array.isArray(config.svg.props.children)) {
        config.svg = config.svg.props.children;
    }
    if (Array.isArray(config.svg)) {
        config.svg = config.svg.map((el) => {
            if (el && typeof el === 'object' && el.tag && !el.type) {
                const { tag, ...rest } = el;
                return { ...rest, type: tag };
            }
            return el;
        });
    }
    if (!config.category || typeof config.category !== 'string') {
        config.category = "Personnalisé";
    }
    if (typeof config.label === 'string' && config.label && typeof config.category === 'string' && config.category && typeof config.width === 'number' && config.width > 0 && typeof config.height === 'number' && config.height > 0 && Array.isArray(config.ports) && config.ports.every((p) => typeof p.x === 'number' && typeof p.y === 'number') && Array.isArray(config.svg) && config.svg.every((el) => (typeof el.type === 'string' || typeof el.tag === 'string') && typeof el.props === 'object')) {
        return config;
    }
    return null;
};

// === From: src/lib/constants.ts ===
const bulbExample = {
    "label": "Ampoule",
    "category": "Sorties",
    "width": 40,
    "height": 40,
    "ports": [{ "x": 0, "y": 20 }, { "x": 40, "y": 20 }],
    "svg": [{ "type": "circle", "props": { "cx": 20, "cy": 20, "r": 12, "fill": "none", "strokeWidth": 2 } }, { "type": "line", "props": { "x1": 0, "y1": 20, "x2": 8, "y2": 20, "strokeWidth": 2 } }, { "type": "line", "props": { "x1": 32, "y1": 20, "x2": 40, "y2": 20, "strokeWidth": 2 } }, { "type": "path", "props": { "d": "M15 15 L25 25 M25 15 L15 25", "stroke": "currentColor", "strokeWidth": 1.5, "fill": "none" } }]
};
const aiQueryTemplate = `Vous êtes un expert en conception de circuits électriques.
Vous devez générer un composant JSON basé sur la demande de l'utilisateur ci-dessous.

---
**Demande de l'utilisateur :**
Un condensateur non polarisé, catégorie Passifs.
---

**RÈGLES CRITIQUES**
1.  **Grille :** La conception est basée sur une grille de 20 unités.
2.  **Dimensions (\`width\`, \`height\`) :** DOIVENT être des multiples de 20.
3.  **Connecteurs (\`ports\`) :** DOIVENT être placés EXACTEMENT sur les bords du composant, avec des coordonnées \`x\` et \`y\` qui sont des multiples de 20.
4.  **Dessin (\`svg\`) :** Les coordonnées et dimensions SVG sont relatives au coin supérieur gauche (0,0) du composant.
5.  **Catégorie (\`category\`) :** La catégorie DOIT être l'une des suivantes : Sources, Passifs, Commutation, Sorties, Annotations, Personnalisé. Si la demande n'est pas claire, utilisez "Personnalisé".

---

Veuillez générer la sortie sous la forme d'un unique bloc de code JSON. Ne pas inclure de texte avant ou après le bloc de code. L'objet doit respecter cette structure (ceci est un exemple pour une ampoule, adaptez-le à la demande de l'utilisateur) :

{
  "label": "Ampoule",
  "category": "Sorties",
  "width": 40,
  "height": 40,
  "ports": [
    {
      "x": 0,
      "y": 20
    },
    {
      "x": 40,
      "y": 20
    }
  ],
  "svg": [
    {
      "type": "circle",
      "props": {
        "cx": 20,
        "cy": 20,
        "r": 12,
        "fill": "none",
        "strokeWidth": 2
      }
    },
    {
      "type": "line",
      "props": {
        "x1": 0,
        "y1": 20,
        "x2": 8,
        "y2": 20,
        "strokeWidth": 2
      }
    },
    {
      "type": "line",
      "props": {
        "x1": 32,
        "y1": 20,
        "x2": 40,
        "y2": 20,
        "strokeWidth": 2
      }
    },
    {
      "type": "path",
      "props": {
        "d": "M15 15 L25 25 M25 15 L15 25",
        "stroke": "currentColor",
        "strokeWidth": 1.5,
        "fill": "none"
      }
    }
  ]
}`;

// === From: src/components.ts ===
const BUILTIN_COMPONENT_CONFIG = {
    battery: {
        label: 'Batterie',
        category: 'Sources',
        width: 80,
        height: 40,
        ports: [{ x: 0, y: 20 }, { x: 80, y: 20 }],
        svg: React.createElement(React.Fragment, null, React.createElement('path', { d: "M26 5 V35 M35 10 V30 M44 5 V35 M53 10 V30", stroke: "currentColor", strokeWidth: "2", fill: "none" }), React.createElement('line', { x1: "0", y1: "20", x2: "26", y2: "20", stroke: "currentColor", strokeWidth: "2" }), React.createElement('line', { x1: "53", y1: "20", x2: "80", y2: "20", stroke: "currentColor", strokeWidth: "2" }), React.createElement('text', { x: "5", y: "12", fontSize: "12", fill: "currentColor" }, "+"), React.createElement('text', { x: "70", y: "12", fontSize: "12", fill: "currentColor" }, "-")),
    },
    resistor: {
        label: 'Résistance',
        category: 'Passifs',
        width: 80,
        height: 40,
        ports: [{ x: 0, y: 20 }, { x: 80, y: 20 }],
        svg: React.createElement(React.Fragment, null, React.createElement('line', { x1: "0", y1: "20", x2: "20", y2: "20", stroke: "currentColor", strokeWidth: "2" }), React.createElement('path', { d: "M 20 20 L 25 10 L 31 30 L 37 10 L 43 30 L 49 10 L 55 30 L 60 20", stroke: "currentColor", fill: "none", strokeWidth: "2" }), React.createElement('line', { x1: "60", y1: "20", x2: "80", y2: "20", stroke: "currentColor", strokeWidth: "2" })),
    },
    switch: {
        label: 'Interrupteur SPST',
        category: 'Commutation',
        width: 80,
        height: 40,
        ports: [{ x: 0, y: 20 }, { x: 80, y: 20 }],
        svg: React.createElement(React.Fragment, null, React.createElement('circle', { cx: "10", cy: "20", r: "3", fill: "currentColor" }), React.createElement('circle', { cx: "70", cy: "20", r: "3", fill: "currentColor" }), React.createElement('line', { x1: "0", y1: "20", x2: "10", y2: "20", stroke: "currentColor", strokeWidth: "2" }), React.createElement('line', { x1: "12", y1: "20", x2: "60", y2: "10", stroke: "currentColor", strokeWidth: "2" }), React.createElement('line', { x1: "70", y1: "20", x2: "80", y2: "20", stroke: "currentColor", strokeWidth: "2" })),
    },
    switch_spdt: {
        label: 'Interrupteur SPDT',
        category: 'Commutation',
        width: 80,
        height: 80,
        ports: [{ x: 0, y: 40 }, { x: 80, y: 20 }, { x: 80, y: 60 }],
        svg: React.createElement(React.Fragment, null, React.createElement('circle', { cx: "10", cy: "40", r: "3", fill: "currentColor" }), React.createElement('circle', { cx: "70", cy: "20", r: "3", fill: "currentColor" }), React.createElement('circle', { cx: "70", cy: "60", r: "3", fill: "currentColor" }), React.createElement('line', { x1: "0", y1: "40", x2: "10", y2: "40", stroke: "currentColor", strokeWidth: "2" }), React.createElement('line', { x1: "70", y1: "20", x2: "80", y2: "20", stroke: "currentColor", strokeWidth: "2" }), React.createElement('line', { x1: "70", y1: "60", x2: "80", y2: "60", stroke: "currentColor", strokeWidth: "2" }), React.createElement('line', { x1: "10", y1: "40", x2: "70", y2: "20", stroke: "currentColor", strokeWidth: "2" })),
    },
    led: {
        label: 'DEL',
        category: 'Sorties',
        width: 40,
        height: 40,
        ports: [{ x: 0, y: 20 }, { x: 40, y: 20 }],
        svg: React.createElement(React.Fragment, null, React.createElement('path', { d: "M10 5 L30 20 L10 35 Z", stroke: "currentColor", strokeWidth: "2", fill: "none" }), React.createElement('line', { x1: "10", y1: "20", x2: "0", y2: "20", stroke: "currentColor", strokeWidth: "2" }), React.createElement('line', { x1: "30", y1: "20", x2: "40", y2: "20", stroke: "currentColor", strokeWidth: "2" }), React.createElement('line', { x1: "30", y1: "5", x2: "30", y2: "35", stroke: "currentColor", strokeWidth: "2" }), React.createElement('g', { transform: "translate(5, 5)" }, React.createElement('line', { x1: "25", y1: "0", x2: "35", y2: "-10", stroke: "currentColor", strokeWidth: "1" }), React.createElement('line', { x1: "30", y1: "0", x2: "40", y2: "-10", stroke: "currentColor", strokeWidth: "1" }), React.createElement('path', { d: "M33 -12 L35 -10 L37 -12", stroke: "currentColor", strokeWidth: "1", fill: "none" }), React.createElement('path', { d: "M38 -12 L40 -10 L42 -12", stroke: "currentColor", strokeWidth: "1", fill: "none" }))),
    },
    text: {
        label: 'Titre',
        category: 'Annotations',
        width: 80,
        height: 40,
        ports: [{ x: 0, y: 20 }, { x: 80, y: 20 }, { x: 40, y: 0 }, { x: 40, y: 40 }],
        svg: React.createElement(React.Fragment, null),
    },
    node: {
        label: 'Nœud',
        category: 'Système',
        width: 10,
        height: 10,
        ports: [{ x: 5, y: 5 }],
        svg: React.createElement('circle', { cx: 5, cy: 5, r: 5, fill: "black" })
    }
};

// === From: src/components/RenderCustomSvg.tsx ===
const RenderCustomSvg = ({ elements }) => {
    const renderElements = (els) => {
        return els.map((el, index) => {
            if (el.type === 'arc' && el.props) {
                const { x1, y1, x2, y2, rx, ry, largeArcFlag, sweepFlag, ...restProps } = el.props;
                const pathData = `M ${x1} ${y1} A ${rx} ${ry} 0 ${largeArcFlag || 0} ${sweepFlag || 0} ${x2} ${y2}`;
                const pathProps = {
                    ...restProps,
                    d: pathData,
                    key: index,
                };
                if (!pathProps.stroke) {
                    pathProps.stroke = 'currentColor';
                }
                return React.createElement('path', pathProps);
            }
            const children = el.children ? renderElements(el.children) : el.textContent || null;
            const props = { ...el.props,
                key: index
            };
            if (!props.stroke && (el.type === 'path' || el.type === 'line' || el.type === 'rect' || el.type === 'circle')) {
                props.stroke = 'currentColor';
            }
            return React.createElement(el.type, props, children);
        });
    };
    return React.createElement(React.Fragment, null, ...renderElements(elements));
};

// === From: src/components/WireRenderer.tsx ===
const WireRenderer = React.memo(({
    wire,
    isSelected,
    onClick,
    onDoubleClick,
    onDelete,
    onHandleMouseDown,
    onHandleContextMenu,
}) => {
    return (React.createElement("g", null, React.createElement("path", {
        d: generateOrthogonalPathString(wire.path),
        stroke: "transparent",
        strokeWidth: "20",
        fill: "none",
        className: "cursor-pointer",
        onClick: (e) => onClick(e, wire.id),
        onDoubleClick: (e) => onDoubleClick(e, wire.id)
    }), React.createElement("path", {
        d: generateOrthogonalPathString(wire.path),
        stroke: "black",
        strokeWidth: isSelected ? 4 : 2,
        fill: "none",
        className: "pe-none transition-all"
    }), isSelected && wire.path.slice(1, -1).map((point, index) => {
        const pointIndex = index + 1;
        return (React.createElement("circle", {
            key: `${wire.id}-handle-${pointIndex}`,
            cx: point.x,
            cy: point.y,
            r: "6",
            fill: "black",
            stroke: "white",
            strokeWidth: "2",
            className: "cursor-grab active:cursor-grabbing",
            onMouseDown: (e) => onHandleMouseDown(e, wire.id, pointIndex),
            onContextMenu: (e) => onHandleContextMenu(e, wire.id, pointIndex)
        }, React.createElement("title", null, "Faire glisser pour déplacer. Clic-droit pour supprimer.")));
    }), isSelected && (() => {
        if (!wire || wire.path.length < 1)
            return null;
        const midIndex = Math.floor(wire.path.length / 2);
        const p1 = wire.path[midIndex - 1] || wire.path[0];
        const p2 = wire.path[midIndex] || p1;
        const midPoint = { x: (p1.x + p2.x) / 2, y: (p1.y + p2.y) / 2 };
        return (React.createElement("g", {
            className: "cursor-pointer",
            onClick: (e) => {
                e.stopPropagation();
                onDelete(wire.id);
            },
            transform: `translate(${midPoint.x}, ${midPoint.y})`
        }, React.createElement("title", null, "Supprimer le fil"), React.createElement("circle", { r: "12", fill: "rgba(239, 68, 68, 0.8)", stroke: "white", strokeWidth: "1" }), React.createElement("text", { y: "5", textAnchor: "middle", fill: "white", fontSize: "14", className: "pointer-events-none" }, "🗑️")));
    })()));
});

// === From: src/components/Minimap.tsx ===
const Minimap = React.memo(({
    components,
    wires,
    viewBox,
    circuitBounds,
    componentConfig,
    onDragStart,
    minimapRef
}) => {
    return (React.createElement("div", { className: "position-absolute bottom-0 end-0 m-3 rounded-3 shadow-lg overflow-hidden", style: { width: '14rem', height: '11rem', backgroundColor: 'rgba(255,255,255,0.9)' } }, React.createElement("svg", { ref: minimapRef, className: "w-100 h-100", viewBox: `${circuitBounds.minX} ${circuitBounds.minY} ${circuitBounds.width} ${circuitBounds.height}` }, components.map(comp => {
        const config = componentConfig[comp.type];
        if (!config)
            return null;
        const width = comp.width ?? config.width;
        const height = comp.height ?? config.height;
        return React.createElement("rect", { key: comp.id, x: comp.x, y: comp.y, width: width, height: height, fill: comp.type === 'node' ? "black" : "#38bdf8", opacity: "0.7" });
    }), wires.map(wire => (React.createElement("path", { key: wire.id, d: generateOrthogonalPathString(wire.path), stroke: "black", strokeWidth: Math.max(5, circuitBounds.width / 200), fill: "none", opacity: "0.6" }))), React.createElement("rect", {
        x: viewBox.x,
        y: viewBox.y,
        width: viewBox.width,
        height: viewBox.height,
        fill: "rgba(0, 180, 255, 0.2)",
        stroke: "rgba(0, 180, 255, 0.9)",
        strokeWidth: Math.max(8, circuitBounds.width / 100),
        className: "cursor-grab",
        onMouseDown: onDragStart
    }))));
});

// === From: src/components/ComponentRenderer.tsx ===
const ComponentRenderer = React.memo(({
    comp,
    config,
    isSelected,
    isEditing,
    isComponentConnected,
    isExporting,
    onMouseDown,
    onUpdateComponent,
    onTitleBlur,
    onDoubleClick,
    onRotate,
    onDelete,
    onPortMouseDown,
    onPortMouseUp,
    getTitleProps,
    drawingWire
}) => {
    if (comp.type === 'text') {
        const textMeasureRef = React.useRef(null);
        const inputRef = React.useRef(null);
        React.useLayoutEffect(() => {
            if (isEditing) {
                inputRef.current?.focus();
                inputRef.current?.select();
            }
        }, [isEditing]);
        const handleTextBlur = (e) => {
            const newTitle = e.target.value;
            if (textMeasureRef.current) {
                textMeasureRef.current.textContent = newTitle || ' '; // Use a space for empty string to correctly measure height
                const bbox = textMeasureRef.current.getBBox();
                const padding = 20;
                // Ensure dimensions are multiples of 40 (GRID_SIZE * 2) so centered ports align with the grid.
                const newWidth = Math.max(config.width, Math.ceil((bbox.width + padding) / (GRID_SIZE * 2)) * (GRID_SIZE * 2));
                const newHeight = Math.max(config.height, Math.ceil((bbox.height + padding) / (GRID_SIZE * 2)) * (GRID_SIZE * 2));
                onUpdateComponent(comp.id, { title: newTitle, width: newWidth, height: newHeight });
            } else {
                onUpdateComponent(comp.id, { title: newTitle });
            }
            onTitleBlur();
        };
        const width = comp.width ?? config.width;
        const height = comp.height ?? config.height;
        const ports = [
            { x: 0, y: height / 2 },
            { x: width, y: height / 2 },
            { x: width / 2, y: 0 },
            { x: width / 2, y: height }
        ];
        return (React.createElement("g", {
            transform: `translate(${comp.x}, ${comp.y})`,
            "aria-label": `Annotation titre ${comp.title}`
        }, React.createElement("text", { ref: textMeasureRef, x: "-9999", y: "-9999", style: { fontSize: '18px', userSelect: 'none', pointerEvents: 'none' } }, comp.title), isSelected && !isExporting && !isEditing && (React.createElement("rect", {
            x: "-4",
            y: "-4",
            width: width + 8,
            height: height + 8,
            fill: "none",
            stroke: "#38bdf8",
            strokeWidth: "1",
            rx: "4",
            className: "pe-none"
        })), React.createElement("g", { transform: `rotate(${comp.rotation}, ${width / 2}, ${height / 2})` }, isEditing && !isExporting ? (React.createElement("foreignObject", { width: width, height: height, className: "overflow-visible" }, React.createElement("div", { className: "w-100 h-100 d-flex align-items-center justify-content-center" }, React.createElement("input", {
            ref: inputRef,
            type: "text",
            defaultValue: comp.title,
            onBlur: handleTextBlur,
            onMouseDown: (e) => e.stopPropagation(),
            onClick: (e) => e.stopPropagation(),
            onKeyDown: (e) => {
                if (e.key === 'Enter')
                    e.currentTarget.blur();
            },
            className: "form-control form-control-lg text-center h-100",
            "aria-label": "Contenu du texte"
        })))) : (React.createElement("g", { onDoubleClick: (e) => {
                e.stopPropagation();
                if (!isExporting)
                    onDoubleClick(comp.id);
            } }, React.createElement("rect", {
            width: width,
            height: height,
            fill: "transparent",
            stroke: isComponentConnected ? "black" : "none",
            strokeWidth: "2",
            rx: "2",
            className: isExporting ? "" : "cursor-grab",
            onMouseDown: isExporting ? undefined : (e) => onMouseDown(e, comp.id)
        }), React.createElement("text", {
            x: width / 2,
            y: height / 2,
            fontSize: "18",
            fill: "black",
            dominantBaseline: "middle",
            textAnchor: "middle",
            className: "pe-none"
        }, comp.title))), !isExporting && ports.map((port, index) => {
            if (!drawingWire) {
                return null;
            }
            // Calculate absolute port position to check distance
            const centerX = width / 2;
            const centerY = height / 2;
            const translatedX = port.x - centerX;
            const translatedY = port.y - centerY;
            const rad = (comp.rotation || 0) * Math.PI / 180;
            const cos = Math.cos(rad);
            const sin = Math.sin(rad);
            const rotatedX = translatedX * cos - translatedY * sin;
            const rotatedY = translatedX * sin + translatedY * cos;
            const absPortX = comp.x + rotatedX + centerX;
            const absPortY = comp.y + rotatedY + centerY;
            const distSq = Math.pow(drawingWire.currentPos.x - absPortX, 2) + Math.pow(drawingWire.currentPos.y - absPortY, 2);
            const proximityThreshold = GRID_SIZE * 4; // 80px
            if (distSq > proximityThreshold * proximityThreshold) {
                return null;
            }
            return (React.createElement("circle", {
                key: index,
                cx: port.x,
                cy: port.y,
                r: "5",
                fill: "black",
                className: "cursor-pointer port-handle",
                onMouseDown: (e) => onPortMouseDown(e, comp.id, index),
                onMouseUp: (e) => onPortMouseUp(e, comp.id, index),
                "aria-label": `Port ${index + 1}`
            }));
        })), isSelected && !isExporting && (React.createElement("g", null, React.createElement("g", { className: "cursor-pointer", onClick: (e) => {
            e.stopPropagation();
            onRotate(comp.id);
        } }, React.createElement("title", null, "Pivoter le composant"), React.createElement("circle", { cx: width + 15, cy: height / 2 - 10, r: "10", fill: "#2563eb" }), React.createElement("text", { x: width + 15, y: height / 2 - 5, fontSize: "14", fill: "white", textAnchor: "middle" }, "↻")), React.createElement("g", { className: "cursor-pointer", onClick: (e) => {
            e.stopPropagation();
            onDelete(comp.id);
        } }, React.createElement("title", null, "Supprimer le composant"), React.createElement("circle", { cx: width + 15, cy: height / 2 + 10, r: "10", fill: "#dc2626" }), React.createElement("text", { x: width + 15, y: height / 2 + 15, fontSize: "14", fill: "white", textAnchor: "middle" }, "🗑️"))))));
    }
    const { foProps, textProps } = getTitleProps(comp);
    return (React.createElement("g", {
        transform: `translate(${comp.x}, ${comp.y})`,
        "aria-label": `Composant ${comp.title}`
    }, isSelected && comp.type !== 'node' && (React.createElement("rect", {
        x: "-4",
        y: "-4",
        width: config.width + 8,
        height: config.height + 8,
        fill: "none",
        stroke: "#38bdf8",
        strokeWidth: "1",
        rx: "4",
        className: "pe-none"
    })), React.createElement("g", { transform: `rotate(${comp.rotation}, ${config.width / 2}, ${config.height / 2})` }, isExporting && React.createElement("rect", { width: config.width, height: config.height, fill: "white" }), React.createElement("rect", {
        width: config.width,
        height: config.height,
        fill: "transparent",
        className: isExporting ? "" : "cursor-grab",
        onMouseDown: isExporting ? undefined : (e) => onMouseDown(e, comp.id),
        onClick: isExporting ? undefined : (e) => e.stopPropagation()
    }), React.createElement("g", { className: comp.type !== 'node' ? "text-black" : "", pointerEvents: "none" }, React.isValidElement(config.svg) ? config.svg : React.createElement(RenderCustomSvg, { elements: config.svg })), !isExporting && config.ports.map((port, index) => (React.createElement("circle", {
        key: index,
        cx: port.x,
        cy: port.y,
        r: "5",
        fill: "black",
        className: "cursor-pointer port-handle",
        onMouseDown: (e) => onPortMouseDown(e, comp.id, index),
        onMouseUp: (e) => onPortMouseUp(e, comp.id, index),
        "aria-label": `Port ${index + 1}`
    })))), isSelected && comp.type !== 'node' ? (React.createElement("foreignObject", { ...foProps }, React.createElement("input", {
        type: "text",
        value: comp.title,
        onChange: (e) => onUpdateComponent(comp.id, { title: e.target.value }),
        onBlur: onTitleBlur,
        onMouseDown: (e) => e.stopPropagation(),
        onClick: (e) => e.stopPropagation(),
        className: `form-control form-control-sm ${textProps.textAnchor === 'middle' ? 'text-center' : textProps.textAnchor === 'start' ? 'text-start' : 'text-end'}`,
        "aria-label": "Titre du composant"
    }))) : comp.type !== 'node' ? (React.createElement("text", {
        x: textProps.x,
        y: textProps.y,
        fontSize: "12",
        fill: "black",
        textAnchor: textProps.textAnchor,
        className: "pe-none"
    }, comp.title)) : null, isSelected && comp.type !== 'node' && (React.createElement("g", null, React.createElement("g", { className: "cursor-pointer", onClick: (e) => {
        e.stopPropagation();
        onRotate(comp.id);
    } }, React.createElement("title", null, "Pivoter le composant"), React.createElement("circle", { cx: config.width + 15, cy: config.height / 2 - 10, r: "10", fill: "#2563eb" }), React.createElement("text", { x: config.width + 15, y: config.height / 2 - 5, fontSize: "14", fill: "white", textAnchor: "middle" }, "↻")), React.createElement("g", { className: "cursor-pointer", onClick: (e) => {
        e.stopPropagation();
        onDelete(comp.id);
    } }, React.createElement("title", null, "Supprimer le composant"), React.createElement("circle", { cx: config.width + 15, cy: config.height / 2 + 10, r: "10", fill: "#dc2626" }), React.createElement("text", { x: config.width + 15, y: config.height / 2 + 15, fontSize: "14", fill: "white", textAnchor: "middle" }, "🗑️"))))));
});

// === From: src/components/Sidebar.tsx ===
class Sidebar extends React.Component {
    constructor(props) {
        super(props);
        this.state = {
            searchTerm: '',
            expandedCategories: {},
            isMenuOpen: false,
        };
        this.menuRef = React.createRef();
        this.handleClickOutside = this.handleClickOutside.bind(this);
        this.toggleCategory = this.toggleCategory.bind(this);
    }
    componentDidMount() {
        document.addEventListener("mousedown", this.handleClickOutside);
    }
    componentWillUnmount() {
        document.removeEventListener("mousedown", this.handleClickOutside);
    }
    handleClickOutside(event) {
        if (this.state.isMenuOpen && this.menuRef.current && !this.menuRef.current.contains(event.target)) {
            this.setState({ isMenuOpen: false });
        }
    }
    toggleCategory(category) {
        this.setState(prevState => ({
            expandedCategories: {
                ...prevState.expandedCategories,
                [category]: !(prevState.expandedCategories[category] ?? true),
            }
        }));
    }
    getCategorizedComponents() {
        const { componentConfig } = this.props;
        const { searchTerm } = this.state;
        const searchLower = searchTerm.toLowerCase();
        const filtered = Object.entries(componentConfig)
            .filter(([type, config]) => type !== 'node' &&
            (config.label.toLowerCase().includes(searchLower) ||
                config.category.toLowerCase().includes(searchLower)));
        return filtered.reduce((acc, [type, config]) => {
            const category = config.category || 'Autres';
            if (!acc[category]) {
                acc[category] = [];
            }
            acc[category].push({ type: type, config });
            return acc;
        }, {});
    }
    render() {
        const { onDragStart, onSave, onLoad, onReset, onManageComponents, onUndo, onRedo, canUndo, canRedo, onGenerateImage, onCopyImage, } = this.props;
        const { searchTerm, isMenuOpen, expandedCategories } = this.state;
        const categorizedComponents = this.getCategorizedComponents();
        return (React.createElement("aside", { className: "bg-light border-end p-3 d-flex flex-column gap-3 shadow z-1", style: { width: '16rem' } }, React.createElement("div", { className: "d-flex justify-content-between align-items-center position-relative" }, React.createElement("h1", { className: "h4 fw-bold text-primary" }, "Composants"), React.createElement("button", {
            onClick: () => this.setState(prevState => ({ isMenuOpen: !prevState.isMenuOpen })),
            className: "btn btn-light rounded-circle p-2",
            "aria-haspopup": "true",
            "aria-expanded": isMenuOpen,
            "aria-label": "Ouvrir le menu d'options"
        }, React.createElement("svg", { xmlns: "http://www.w3.org/2000/svg", viewBox: "0 0 20 20", fill: "currentColor", style: { height: 20, width: 20 } }, React.createElement("path", { d: "M10 6a2 2 0 110-4 2 2 0 010 4zM10 12a2 2 0 110-4 2 2 0 010 4zM10 18a2 2 0 110-4 2 2 0 010 4z" }))), isMenuOpen && (React.createElement("div", { ref: this.menuRef, className: "position-absolute end-0 top-100 mt-2 bg-white rounded shadow-lg z-2 animate-fade-in border", style: { width: '18rem' } }, React.createElement("div", { className: "p-2", role: "menu", "aria-orientation": "vertical", "aria-labelledby": "options-menu" }, React.createElement("button", { onClick: () => {
            onManageComponents();
            this.setState({ isMenuOpen: false });
        }, className: "dropdown-item d-flex align-items-center gap-3 px-3 py-2", role: "menuitem" }, React.createElement("svg", { xmlns: "http://www.w3.org/2000/svg", viewBox: "0 0 20 20", fill: "currentColor", style: { width: '1rem', height: '1rem', opacity: 0.7 } }, React.createElement("path", { fillRule: "evenodd", d: "M11.078 2.25c-.217-1.203-1.936-1.203-2.154 0l-.294 1.623a1.75 1.75 0 00-1.058 1.058l-1.623.294c-1.203.217-1.203 1.936 0 2.154l1.623.294a1.75 1.75 0 001.058 1.058l.294 1.623c.217 1.203 1.936 1.203 2.154 0l.294-1.623a1.75 1.75 0 001.058-1.058l1.623-.294c1.203-.217 1.203-1.936 0-2.154l-1.623-.294a1.75 1.75 0 00-1.058-1.058l-.294-1.623zM10 5.25a.75.75 0 01.75.75v.008a.75.75 0 01-.75.75h-.008a.75.75 0 01-.75-.75v-.008c0-.414.336-.75.75-.75h.008zM10 8.5a.75.75 0 01.75.75v5.008a.75.75 0 01-.75.75h-.008a.75.75 0 01-.75-.75V9.25a.75.75 0 01.75-.75h.008z", clipRule: "evenodd" })), "Gérer les composants"), React.createElement("button", { onClick: () => {
            onSave();
            this.setState({ isMenuOpen: false });
        }, className: "dropdown-item d-flex align-items-center gap-3 px-3 py-2", role: "menuitem" }, React.createElement("svg", { xmlns: "http://www.w3.org/2000/svg", viewBox: "0 0 20 20", fill: "currentColor", style: { width: '1rem', height: '1rem', opacity: 0.7 } }, React.createElement("path", { d: "M10.75 2.75a.75.75 0 00-1.5 0v8.614L6.295 8.235a.75.75 0 10-1.09 1.03l4.25 4.5a.75.75 0 001.09 0l4.25-4.5a.75.75 0 00-1.09-1.03l-2.955 3.129V2.75z" }), React.createElement("path", { d: "M3.5 12.75a.75.75 0 00-1.5 0v2.5A2.75 2.75 0 004.75 18h10.5A2.75 2.75 0 0018 15.25v-2.5a.75.75 0 00-1.5 0v2.5c0 .69-.56 1.25-1.25 1.25H4.75c-.69 0-1.25-.56-1.25-1.25v-2.5z" })), "Sauvegarder"), React.createElement("button", { onClick: () => {
            onLoad();
            this.setState({ isMenuOpen: false });
        }, className: "dropdown-item d-flex align-items-center gap-3 px-3 py-2", role: "menuitem" }, React.createElement("svg", { xmlns: "http://www.w3.org/2000/svg", viewBox: "0 0 20 20", fill: "currentColor", style: { width: '1rem', height: '1rem', opacity: 0.7 } }, React.createElement("path", { d: "M10.75 4.75a.75.75 0 00-1.5 0v4.51L6.53 7.53a.75.75 0 00-1.06 1.06l3.25 3.25a.75.75 0 001.06 0l3.25-3.25a.75.75 0 10-1.06-1.06L10.75 9.26V4.75z" }), React.createElement("path", { d: "M3.5 9.75a.75.75 0 00-1.5 0v4.5A2.75 2.75 0 004.75 17h10.5A2.75 2.75 0 0018 14.25v-4.5a.75.75 0 00-1.5 0v4.5c0 .69-.56 1.25-1.25 1.25H4.75c-.69 0-1.25-.56-1.25-1.25v-4.5z" })), "Charger"), React.createElement("button", { onClick: () => {
            onGenerateImage();
            this.setState({ isMenuOpen: false });
        }, className: "dropdown-item d-flex align-items-center gap-3 px-3 py-2", role: "menuitem" }, React.createElement("svg", { xmlns: "http://www.w3.org/2000/svg", viewBox: "0 0 20 20", fill: "currentColor", style: { width: '1rem', height: '1rem', opacity: 0.7 } }, React.createElement("path", { d: "M5.25 5.5a.75.75 0 00-1.5 0v1.25a.75.75 0 001.5 0V5.5zM3 3.5A1.5 1.5 0 014.5 2h11A1.5 1.5 0 0117 3.5v11a1.5 1.5 0 01-1.5 1.5h-11A1.5 1.5 0 013 14.5v-11zM4.5 3.5a.5.5 0 00-.5.5v11c0 .28.22.5.5.5h11a.5.5 0 00.5-.5v-11a.5.5 0 00-.5-.5h-11z" }), React.createElement("path", { d: "M8.25 7a.75.75 0 00-1.5 0v.25a.75.75 0 001.5 0V7zM9 10.5a1.5 1.5 0 11-3 0 1.5 1.5 0 013 0z" }), React.createElement("path", { d: "M11.75 7.5a.75.75 0 00-1.5 0v5a.75.75 0 001.5 0v-5z" }), React.createElement("path", { d: "M14.25 7a.75.75 0 00-1.5 0v.25a.75.75 0 001.5 0V7zM15 10.5a1.5 1.5 0 11-3 0 1.5 1.5 0 013 0z" })), "Télécharger l'image"), React.createElement("button", { onClick: () => {
            onCopyImage();
            this.setState({ isMenuOpen: false });
        }, className: "dropdown-item d-flex align-items-center gap-3 px-3 py-2", role: "menuitem" }, React.createElement("svg", { xmlns: "http://www.w3.org/2000/svg", viewBox: "0 0 20 20", fill: "currentColor", style: { width: '1rem', height: '1rem', opacity: 0.7 } }, React.createElement("path", { d: "M7 3.5A1.5 1.5 0 018.5 2h5.879a1.5 1.5 0 011.06.44l3.122 3.121a1.5 1.5 0 01.439 1.061V16.5A1.5 1.5 0 0117.5 18h-9A1.5 1.5 0 017 16.5v-13z" }), React.createElement("path", { d: "M5 4.5A1.5 1.5 0 016.5 3H10v2.75A1.25 1.25 0 0011.25 7H14v9.5A1.5 1.5 0 0112.5 18h-8A1.5 1.5 0 013 16.5v-10A1.5 1.5 0 014.5 5H5v-.5z" })), "Copier l'image"), React.createElement("div", { className: "border-top my-1" }), React.createElement("button", { onClick: () => {
            onReset();
            this.setState({ isMenuOpen: false });
        }, className: "dropdown-item text-danger d-flex align-items-center gap-3 px-3 py-2", role: "menuitem" }, React.createElement("svg", { xmlns: "http://www.w3.org/2000/svg", viewBox: "0 0 20 20", fill: "currentColor", style: { width: '1rem', height: '1rem', opacity: 0.7 } }, React.createElement("path", { fillRule: "evenodd", d: "M8.75 1A2.75 2.75 0 006 3.75v.443c-.795.077-1.58.22-2.365.468a.75.75 0 10.23 1.482l.149-.022a.75.75 0 01.764.764v8.5c0 1.517 1.233 2.75 2.75 2.75h3.5a2.75 2.75 0 002.75-2.75v-8.5a.75.75 0 01.764-.764l.149.022a.75.75 0 10.23-1.482A41.03 41.03 0 0014 4.193V3.75A2.75 2.75 0 0011.25 1h-2.5zM10 4c.84 0 1.673.025 2.5.075V3.75c0-.69-.56-1.25-1.25-1.25h-2.5c-.69 0-1.25.56-1.25 1.25v.325C8.327 4.025 9.16 4 10 4zM8.58 7.72a.75.75 0 00-1.5.06l.3 7.5a.75.75 0 101.5-.06l-.3-7.5zm4.34.06a.75.75 0 10-1.5-.06l-.3 7.5a.75.75 0 101.5.06l.3-7.5z", clipRule: "evenodd" })), "Effacer"))))), React.createElement("div", { className: "d-flex gap-2" }, React.createElement("button", {
            onClick: onUndo,
            disabled: !canUndo,
            className: "w-100 btn btn-light fw-bold fs-5",
            title: "Annuler (Ctrl+Z)",
            "aria-label": "Annuler la dernière action"
        }, "⟲"), React.createElement("button", {
            onClick: onRedo,
            disabled: !canRedo,
            className: "w-100 btn btn-light fw-bold fs-5",
            title: "Rétablir (Ctrl+Y)",
            "aria-label": "Rétablir la dernière action"
        }, "⟳")), React.createElement("div", { className: "position-relative" }, React.createElement("span", { className: "position-absolute top-50 start-0 translate-middle-y ps-3 pe-none" }, React.createElement("svg", { className: "text-muted", style: { width: '20px', height: '20px' }, viewBox: "0 0 24 24", fill: "none", stroke: "currentColor", strokeWidth: "2", strokeLinecap: "round", strokeLinejoin: "round" }, React.createElement("circle", { cx: "11", cy: "11", r: "8" }), React.createElement("line", { x1: "21", y1: "21", x2: "16.65", y2: "16.65" }))), React.createElement("input", {
            type: "text",
            placeholder: "Rechercher...",
            value: searchTerm,
            onChange: (e) => this.setState({ searchTerm: e.target.value }),
            className: "form-control ps-5"
        })), React.createElement("div", { className: "flex-grow-1 overflow-y-auto pe-2" }, React.createElement("div", { className: "d-flex flex-column gap-1" }, Object.entries(categorizedComponents).sort(([catA], [catB]) => catA.localeCompare(catB)).map(([category, componentsList]) => {
            const isExpanded = searchTerm.length > 0 || (expandedCategories[category] !== false);
            return (React.createElement("div", { key: category }, React.createElement("button", {
                onClick: () => this.toggleCategory(category),
                className: "btn btn-light w-100 d-flex justify-content-between align-items-center text-start p-2"
            }, React.createElement("span", { className: "fw-semibold" }, category), React.createElement("span", { style: { transition: 'transform 0.2s' }, className: `small ${isExpanded ? 'rotate-180' : ''}` }, "▼")), isExpanded && (React.createElement("div", { className: "pt-1 ps-2 ms-2 border-start border-2 d-flex flex-column gap-1" }, componentsList.map(({ type, config }) => (React.createElement("div", {
                key: type,
                className: "p-1 bg-white border rounded-3 cursor-grab d-flex flex-column align-items-center transition-all animate-fade-in",
                draggable: true,
                onDragStart: (e) => onDragStart(e, type),
                title: `Glisser pour ajouter un(e) ${config.label.toLowerCase()}`
            }, React.createElement("svg", { width: config.width, height: config.height, viewBox: `0 0 ${config.width} ${config.height}`, className: "text-black pe-none" }, React.isValidElement(config.svg) ? config.svg : React.createElement(RenderCustomSvg, { elements: config.svg })), React.createElement("span", { className: "small text-capitalize user-select-none" }, config.label))))))));
        })))));
    }
}

// === From: src/components/ComponentManagerModal.tsx ===
class ComponentManagerModal extends React.Component {
    constructor(props) {
        super(props);
        this.state = {
            newComponentJson: '',
            showHelpTooltip: false,
            previewComponentConfig: null,
            previewTypeKey: null,
        };
        this.handleCopyExample = this.handleCopyExample.bind(this);
        this.handleCopyQueryExample = this.handleCopyQueryExample.bind(this);
        this.handlePreviewComponent = this.handlePreviewComponent.bind(this);
        this.handleConfirmAddComponent = this.handleConfirmAddComponent.bind(this);
        this.handleCancelPreview = this.handleCancelPreview.bind(this);
        this.handleDeleteCustomComponent = this.handleDeleteCustomComponent.bind(this);
        this.handleExportAllComponents = this.handleExportAllComponents.bind(this);
    }
    handleCopyExample() {
        navigator.clipboard.writeText(JSON.stringify(bulbExample, null, 2))
            .then(() => alert('Exemple copié dans le presse-papiers!'))
            .catch(err => {
            console.error('Erreur lors de la copie: ', err);
            alert('La copie a échoué. Veuillez copier manuellement.');
        });
    }
    
    handleCopyQueryExample() {
        navigator.clipboard.writeText(aiQueryTemplate)
            .then(() => alert('Exemple de requête copié dans le presse-papiers!'))
            .catch(err => {
            console.error('Erreur lors de la copie: ', err);
            alert('La copie a échoué. Veuillez copier manuellement.');
        });
    }
    
    handlePreviewComponent() {
        try {
            if (!this.state.newComponentJson.trim()) {
                alert("Veuillez coller la configuration JSON du composant.");
                return;
            }
            const jsonString = this.state.newComponentJson;
            const jsonMatch = jsonString.match(/\{[\s\S]*\}/);
            if (!jsonMatch) {
                alert("Aucun objet JSON valide n'a été trouvé dans le texte fourni.");
                return;
            }
            const extractedJson = jsonMatch[0];
            const parsedJson = JSON.parse(extractedJson);
            const { allComponentConfigs } = this.props;
            const tempLabel = parsedJson.label;
            if (!tempLabel || typeof tempLabel !== 'string') {
                alert("Le composant doit avoir une propriété 'label'.");
                return;
            }
            const typeKey = tempLabel.toLowerCase().replace(/\s/g, '_');
            if (allComponentConfigs[typeKey]) {
                alert(`Un composant avec le type '${typeKey}' existe déjà.`);
                return;
            }
            const validConfig = normalizeAndValidateConfig(parsedJson);
            if (validConfig) {
                this.setState({
                    previewComponentConfig: validConfig,
                    previewTypeKey: typeKey
                });
            } else {
                alert("La structure JSON du composant est invalide. Veuillez vérifier les champs (width, height, ports, svg, etc.).");
            }
        } catch (e) {
            alert("JSON invalide. Assurez-vous que le texte copié contient un objet JSON bien formé.");
            console.error(e);
        }
    }
    handleConfirmAddComponent() {
        const { previewComponentConfig, previewTypeKey } = this.state;
        const { customComponentConfigs, onUpdateCustomComponents } = this.props;
        if (!previewComponentConfig || !previewTypeKey)
            return;
        const newConfigs = { ...customComponentConfigs, [previewTypeKey]: previewComponentConfig };
        onUpdateCustomComponents(newConfigs);
        alert(`Le composant '${previewComponentConfig.label}' a été ajouté avec succès !`);
        this.setState({
            newComponentJson: '',
            previewComponentConfig: null,
            previewTypeKey: null
        });
    }
    handleCancelPreview() {
        this.setState({
            previewComponentConfig: null,
            previewTypeKey: null
        });
    }
    handleDeleteCustomComponent(typeKey) {
        if (window.confirm(`Êtes-vous sûr de vouloir supprimer le composant '${typeKey}' ?`)) {
            const { customComponentConfigs, onUpdateCustomComponents } = this.props;
            const newConfigs = { ...customComponentConfigs };
            delete newConfigs[typeKey];
            onUpdateCustomComponents(newConfigs);
        }
    }
    
    handleExportAllComponents() {
        const { allComponentConfigs } = this.props;
        if (Object.keys(allComponentConfigs).length === 0) {
            alert("Aucun composant à exporter.");
            return;
        }
        const data = JSON.stringify(allComponentConfigs, null, 2);
        const blob = new Blob([data], { type: 'application/json' });
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = 'components_library.json';
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        URL.revokeObjectURL(url);
    }
    
    render() {
        if (!this.props.isOpen)
            return null;
        const { onClose, customComponentConfigs, onImportClick } = this.props;
        const { newComponentJson, showHelpTooltip, previewComponentConfig } = this.state;
        return (React.createElement("div", { className: "position-absolute top-0 start-0 w-100 vh-100 d-flex align-items-center justify-content-center z-2", style: { backgroundColor: 'rgba(0,0,0,0.7)' }, onClick: onClose }, React.createElement("div", { className: "bg-white rounded-3 shadow-lg p-4 w-100 d-flex flex-column", style: { maxWidth: '56rem', maxHeight: '90vh' }, onClick: e => e.stopPropagation() }, React.createElement("h2", { className: "h2 fw-bold text-primary mb-4" }, "Gestionnaire de Composants"), React.createElement("div", { className: "flex-grow-1 overflow-y-auto pe-3" }, React.createElement("h3", { className: "h5 fw-semibold mb-2" }, "Composants Personnalisés"), React.createElement("div", { className: "d-flex gap-2 my-2" }, React.createElement("button", { onClick: onImportClick, className: "btn btn-primary fw-bold btn-sm" }, "Importer..."), React.createElement("button", { onClick: this.handleExportAllComponents, className: "btn btn-light fw-bold btn-sm" }, "Exporter Tout")), React.createElement("div", { className: "d-flex flex-column gap-2" }, Object.entries(customComponentConfigs).length > 0 ? Object.entries(customComponentConfigs).map(([key, config]) => (React.createElement("div", { key: key, className: "bg-light p-2 rounded d-flex justify-content-between align-items-center small" }, React.createElement("div", null, React.createElement("span", { className: "fw-semibold" }, config.label), React.createElement("span", { className: "text-muted ms-2" }, `(${key})`), React.createElement("span", { className: "ms-4 badge bg-primary-subtle text-primary-emphasis rounded-pill" }, config.category)), React.createElement("div", { className: "d-flex align-items-center gap-3" }, React.createElement("button", { onClick: () => this.handleDeleteCustomComponent(key), className: "btn btn-link text-danger text-decoration-none p-0 fw-bold small text-uppercase" }, "Supprimer"))))) : React.createElement("p", { className: "text-muted" }, "Aucun composant personnalisé.")), React.createElement("hr", { className: "my-4" }), React.createElement("h3", { className: "h5 fw-semibold mb-2" }, "Ajouter un Nouveau Composant"), previewComponentConfig ? (React.createElement("div", { className: "bg-light p-3 rounded border my-4 text-center animate-fade-in" }, React.createElement("h4", { className: "h6 fw-semibold mb-2 text-secondary" }, "Aperçu du Composant"), React.createElement("div", { className: "bg-white p-3 d-inline-block border rounded shadow-sm my-2" }, React.createElement("svg", { width: previewComponentConfig.width, height: previewComponentConfig.height, viewBox: `0 0 ${previewComponentConfig.width} ${previewComponentConfig.height}`, className: "text-black" }, React.createElement(RenderCustomSvg, { elements: previewComponentConfig.svg }))), React.createElement("div", { className: "text-start d-inline-block mt-2 align-top ms-4" }, React.createElement("p", { className: "small text-muted" }, React.createElement("strong", { className: "fw-semibold text-dark" }, "Label:"), " ", previewComponentConfig.label), React.createElement("p", { className: "small text-muted" }, React.createElement("strong", { className: "fw-semibold text-dark" }, "Catégorie:"), " ", previewComponentConfig.category), React.createElement("p", { className: "small text-muted" }, React.createElement("strong", { className: "fw-semibold text-dark" }, "Dimensions:"), " ", previewComponentConfig.width, " x ", previewComponentConfig.height)))) : (React.createElement(React.Fragment, null, React.createElement("button", {
            onClick: () => this.setState({ showHelpTooltip: !this.state.showHelpTooltip }),
            className: "btn btn-outline-primary text-start w-100 d-flex justify-content-between align-items-center mb-2",
            title: "Afficher le guide de génération par IA",
            "aria-expanded": this.state.showHelpTooltip
        }, React.createElement("span", null, "Générer un composant à l'aide de l'IA"), React.createElement("span", { style: { transition: 'transform 0.2s' }, className: this.state.showHelpTooltip ? 'rotate-180' : '' }, "▼")), showHelpTooltip && (React.createElement("div", { className: "bg-light-subtle p-3 rounded mb-4 border small animate-fade-in d-flex flex-column gap-3" }, React.createElement("div", null, React.createElement("h4", { className: "fw-semibold text-primary mb-2" }, "Comment générer un composant avec une IA ?"), React.createElement("p", { className: "text-dark mb-2" }, "Pour créer un nouveau composant avec une IA :"), React.createElement("ol", { className: "list-decimal list-inside text-muted d-flex flex-column gap-1 mb-4" }, React.createElement("li", null, React.createElement("strong", null, "Copiez le modèle :"), " Cliquez sur le bouton \"Copier le modèle\" ci-dessous."), React.createElement("li", null, React.createElement("strong", null, "Modifiez la demande :"), " Collez le modèle dans votre IA favorite. Remplacez le texte sous ", React.createElement("code", null, "**Demande de l'utilisateur :**"), " par la description de votre propre composant."), React.createElement("li", null, React.createElement("strong", null, "Utilisez le JSON :"), " Copiez le JSON brut généré par l'IA et collez-le dans le champ de texte en bas de cette fenêtre.")), React.createElement("div", { className: "position-relative bg-dark p-3 rounded text-white" }, React.createElement("pre", { className: "text-light whitespace-pre-wrap small font-monospace" }, React.createElement("code", null, aiQueryTemplate)), React.createElement("button", {
            onClick: this.handleCopyQueryExample,
            className: "btn btn-primary btn-sm position-absolute top-0 end-0 m-2",
            "aria-label": "Copier le modèle de requête"
        }, "Copier le modèle"))), React.createElement("div", null, React.createElement("h4", { className: "fw-semibold text-primary mb-2" }, "Exemple de résultat JSON attendu"), React.createElement("p", { className: "text-muted fst-italic small mb-2" }, "Voici un exemple du type de JSON que l'IA devrait produire. Remarque : Si l'IA utilise la clé ", React.createElement("code", null, "\"tag\""), " au lieu de ", React.createElement("code", null, "\"type\""), " pour les éléments SVG, le système le corrigera automatiquement. La propriété ", React.createElement("code", null, "\"category\""), " est requise."), React.createElement("div", { className: "position-relative bg-dark p-3 rounded" }, React.createElement("pre", { className: "text-info-emphasis whitespace-pre-wrap small font-monospace" }, React.createElement("code", null, JSON.stringify(bulbExample, null, 2))), React.createElement("button", {
            onClick: this.handleCopyExample,
            className: "btn btn-primary btn-sm position-absolute top-0 end-0 m-2",
            "aria-label": "Copier l'exemple JSON"
        }, "Copier le JSON"))))), React.createElement("p", { className: "small text-muted mb-2" }, "Collez la configuration JSON du composant ci-dessous. Le `label` sera utilisé pour générer une clé unique."), React.createElement("textarea", {
            className: "form-control font-monospace",
            style: { height: '12rem' },
            value: newComponentJson,
            onChange: e => this.setState({ newComponentJson: e.target.value }),
            placeholder: `{\n  "label": "Mon Composant",\n  "category": "Ma Catégorie",\n  "width": 60,\n  "height": 60,\n  "ports": [{"x": 0, "y": 30}, {"x": 60, "y": 30}],\n  "svg": [\n    {"type": "rect", "props": {"x": 10, "y": 10, "width": 40, "height": 40, "fill": "none", "strokeWidth": 2}}\n  ]\n}`
        })))), React.createElement("div", { className: "mt-4 d-flex justify-content-end gap-3" }, previewComponentConfig ? (React.createElement(React.Fragment, null, React.createElement("button", { onClick: this.handleCancelPreview, className: "btn btn-secondary fw-bold" }, "Modifier"), React.createElement("button", { onClick: this.handleConfirmAddComponent, className: "btn btn-success fw-bold" }, "Confirmer l'Ajout"))) : (React.createElement(React.Fragment, null, React.createElement("button", { onClick: onClose, className: "btn btn-secondary fw-bold" }, "Fermer"), React.createElement("button", { onClick: this.handlePreviewComponent, className: "btn btn-primary fw-bold" }, "Prévisualiser le Composant")))))));
    }
}

// === From: src/App.tsx ===
class App extends React.Component {
    constructor(props) {
        super(props);
        this.state = {
            components: [],
            wires: [],
            nextId: 1,
            history: [{ components: [], wires: [], nextId: 1 }],
            historyIndex: 0,
            dragging: null,
            drawingWire: null,
            selectedWireId: null,
            selectedComponentId: null,
            editingComponentId: null,
            draggingHandle: null,
            junctionPreview: null,
            baseComponentConfigs: BUILTIN_COMPONENT_CONFIG,
            customComponentConfigs: {},
            isModalOpen: false,
            isPreviewModalOpen: false,
            previewData: null,
            viewBox: { x: 0, y: 0, width: 1, height: 1 },
            isDraggingMinimap: false,
            isDraggingNewComponent: false,
        };
        this.canvasRef = React.createRef();
        this.minimapRef = React.createRef();
        this.fileInputRef = React.createRef();
        this.componentsFileInputRef = React.createRef();
        this.isDraggingWire = false;
        this.minimapDragStartRef = null;
        this.resizeObserver = null;
        Object.getOwnPropertyNames(Object.getPrototypeOf(this))
            .filter(name => name.startsWith('handle') || name.startsWith('get') || name.startsWith('delete') || name.startsWith('create') || name.startsWith('complete') || name.startsWith('push') || name.startsWith('reset') || name.startsWith('update') || name.startsWith('isComponentConnected') || name.startsWith('generate') || name.startsWith('render'))
            .forEach(method => {
            this[method] = this[method].bind(this);
        });
    }
    pushToHistory(stateUpdate) {
        const { history, historyIndex } = this.state;
        const currentState = {
            components: this.state.components,
            wires: this.state.wires,
            nextId: this.state.nextId,
        };
        const newState = { ...currentState, ...stateUpdate };
        const truncatedHistory = history.slice(0, historyIndex + 1);
        this.setState({
            history: [...truncatedHistory, newState],
            historyIndex: truncatedHistory.length,
        });
    }
    resetHistory(newState) {
        this.setState({
            history: [newState],
            historyIndex: 0,
        });
    }
    handleUndo() {
        const { history, historyIndex } = this.state;
        if (historyIndex > 0) {
            const newIndex = historyIndex - 1;
            const prevState = history[newIndex];
            this.setState({
                ...prevState,
                historyIndex: newIndex,
                selectedComponentId: null,
                selectedWireId: null,
            });
        }
    }
    handleRedo() {
        const { history, historyIndex } = this.state;
        if (historyIndex < history.length - 1) {
            const newIndex = historyIndex + 1;
            const nextState = history[newIndex];
            this.setState({
                ...nextState,
                historyIndex: newIndex,
                selectedComponentId: null,
                selectedWireId: null,
            });
        }
    }
    async componentDidMount() {
        let baseConfigs = BUILTIN_COMPONENT_CONFIG;
        try {
            const response = await fetch('./components_library.json');
            if (!response.ok)
                throw new Error('Le fichier de la bibliothèque de composants est introuvable ou une erreur réseau est survenue.');
            const loadedConfigs = await response.json();
            const validFileConfigs = {};
            for (const key in loadedConfigs) {
                const validConfig = normalizeAndValidateConfig(loadedConfigs[key]);
                if (validConfig) {
                    validFileConfigs[key] = validConfig;
                }
            }
            baseConfigs = validFileConfigs;
        } catch (error) {
            console.warn('Impossible de charger components_library.json. Utilisation des composants intégrés.', error);
        }
        let customConfigs = {};
        try {
            const savedCustoms = localStorage.getItem('customComponents');
            if (savedCustoms) {
                const parsed = JSON.parse(savedCustoms);
                for (const key in parsed) {
                    const validConfig = normalizeAndValidateConfig(parsed[key]);
                    if (validConfig) {
                        customConfigs[key] = validConfig;
                    }
                }
            }
        } catch (e) {
            console.error("Échec du chargement des composants personnalisés depuis localStorage", e);
        }
        this.setState({
            baseComponentConfigs: baseConfigs,
            customComponentConfigs: customConfigs
        });
        this.resizeObserver = new ResizeObserver(entries => {
            for (const entry of entries) {
                const { width, height } = entry.contentRect;
                this.setState(prevState => {
                    if (prevState.viewBox.width !== width || prevState.viewBox.height !== height) {
                        return { viewBox: { ...prevState.viewBox, width, height } };
                    }
                    return null;
                });
            }
        });
        if (this.canvasRef.current) {
            const { width, height } = this.canvasRef.current.getBoundingClientRect();
            this.setState({ viewBox: { x: 0, y: 0, width, height } });
            this.resizeObserver.observe(this.canvasRef.current);
        }
        window.addEventListener('keydown', this.handleKeyDown);
    }
    componentWillUnmount() {
        window.removeEventListener('keydown', this.handleKeyDown);
        if (this.resizeObserver && this.canvasRef.current) {
            this.resizeObserver.unobserve(this.canvasRef.current);
        }
    }
    componentDidUpdate(prevProps, prevState) {
        if (prevState.dragging && !this.state.dragging) {
            this.setState(state => ({
                wires: state.wires.map(wire => {
                    const newFromPos = this.getPortPosition(wire.from.componentId, wire.from.portIndex);
                    const newToPos = this.getPortPosition(wire.to.componentId, wire.to.portIndex);
                    if (wire.path.length < 2)
                        return wire;
                    const newPath = [...wire.path];
                    newPath[0] = newFromPos;
                    newPath[newPath.length - 1] = newToPos;
                    return { ...wire, path: newPath };
                }),
            }));
        }
    }
    isComponentConnected(componentId) {
        return this.state.wires.some(w => w.from.componentId === componentId || w.to.componentId === componentId);
    }
    getMousePosition(e) {
        if (!this.canvasRef.current)
            return { x: 0, y: 0 };
        const svg = this.canvasRef.current;
        const point = svg.createSVGPoint();
        point.x = e.clientX;
        point.y = e.clientY;
        const transformedPoint = point.matrixTransform(svg.getScreenCTM()?.inverse());
        return {
            x: transformedPoint.x,
            y: transformedPoint.y,
        };
    }
    getPortPosition(componentId, portIndex) {
        const component = this.state.components.find(c => c.id === componentId);
        if (!component)
            return { x: 0, y: 0 };
        const COMPONENT_CONFIG = { ...this.state.baseComponentConfigs, ...this.state.customComponentConfigs };
        const config = COMPONENT_CONFIG[component.type];
        if (!config)
            return { x: 0, y: 0 };
        const componentWidth = component.width ?? config.width;
        const componentHeight = component.height ?? config.height;
        let port;
        if (component.type === 'text') {
            const ports = [
                { x: 0, y: componentHeight / 2 },
                { x: componentWidth, y: componentHeight / 2 },
                { x: componentWidth / 2, y: 0 },
                { x: componentWidth / 2, y: componentHeight }
            ];
            port = ports[portIndex];
        } else {
            port = config.ports[portIndex];
        }
        if (!port)
            return { x: 0, y: 0 };
        const centerX = componentWidth / 2;
        const centerY = componentHeight / 2;
        const translatedX = port.x - centerX;
        const translatedY = port.y - centerY;
        const rad = component.rotation * Math.PI / 180;
        const cos = Math.cos(rad);
        const sin = Math.sin(rad);
        const rotatedX = translatedX * cos - translatedY * sin;
        const rotatedY = translatedX * sin + translatedY * cos;
        return {
            x: snapToGrid(component.x + rotatedX + centerX),
            y: snapToGrid(component.y + rotatedY + centerY),
        };
    }
    getTitleProps(comp, configs) {
        const COMPONENT_CONFIG = configs || { ...this.state.baseComponentConfigs, ...this.state.customComponentConfigs };
        const config = COMPONENT_CONFIG[comp.type];
        const margin = 5;
        const inputHeight = 25;
        const inputWidth = 80;
        let foProps = { x: 0, y: 0, width: inputWidth, height: inputHeight };
        let textProps = { x: 0, y: 0, textAnchor: 'middle' };
        switch ((comp.rotation || 0) % 360) {
            case 90:
                foProps.x = config.width + margin;
                foProps.y = (config.height - inputHeight) / 2;
                textProps.x = config.width + margin;
                textProps.y = config.height / 2 + 5;
                textProps.textAnchor = 'start';
                break;
            case 180:
                foProps.x = (config.width - inputWidth) / 2;
                foProps.y = -inputHeight - margin;
                textProps.x = config.width / 2;
                textProps.y = -margin - 5;
                textProps.textAnchor = 'middle';
                break;
            case 270:
                foProps.x = -inputWidth - margin;
                foProps.y = (config.height - inputHeight) / 2;
                textProps.x = -margin;
                textProps.y = config.height / 2 + 5;
                textProps.textAnchor = 'end';
                break;
            case 0:
            default:
                foProps.x = (config.width - inputWidth) / 2;
                foProps.y = config.height + margin;
                textProps.x = config.width / 2;
                textProps.y = config.height + 15;
                textProps.textAnchor = 'middle';
                break;
        }
        return { foProps, textProps };
    }
    getCircuitBounds() {
        const { components, wires, viewBox } = this.state;
        const COMPONENT_CONFIG = { ...this.state.baseComponentConfigs, ...this.state.customComponentConfigs };
        if (components.length === 0 && wires.length === 0) {
            return { minX: -200, minY: -200, width: viewBox.width + 400, height: viewBox.height + 400 };
        }
        let minX = Infinity, minY = Infinity, maxX = -Infinity, maxY = -Infinity;
        components.forEach(comp => {
            const config = COMPONENT_CONFIG[comp.type];
            if (!config)
                return;
            const width = comp.width ?? config.width;
            const height = comp.height ?? config.height;
            minX = Math.min(minX, comp.x - 50);
            minY = Math.min(minY, comp.y - 50);
            maxX = Math.max(maxX, comp.x + width + 50);
            maxY = Math.max(maxY, comp.y + height + 50);
        });
        if (components.length === 0) {
            minX = 0;
            minY = 0;
            maxX = viewBox.width;
            maxY = viewBox.height;
        }
        const padding = 200;
        const width = (maxX - minX) + 2 * padding;
        const height = (maxY - minY) + 2 * padding;
        return {
            minX: minX - padding,
            minY: minY - padding,
            width: Math.max(width, viewBox.width),
            height: Math.max(height, viewBox.height),
        };
    }
    createJunction(junction) {
        const { drawingWire, nextId, wires, components } = this.state;
        if (!drawingWire)
            return;
        const COMPONENT_CONFIG = { ...this.state.baseComponentConfigs, ...this.state.customComponentConfigs };
        const NODE_CONFIG = COMPONENT_CONFIG['node'];
        const nodeSize = NODE_CONFIG.width;
        const newId = nextId;
        const newNodeComponent = {
            id: newId, type: 'node', x: junction.point.x - nodeSize / 2, y: junction.point.y - nodeSize / 2, rotation: 0, title: `Nœud ${newId}`,
        };
        const intersectedWire = wires.find(w => w.id === junction.wireId);
        if (!intersectedWire)
            return;
        const newWireFromDrawing = {
            id: `${drawingWire.from.componentId}:${drawingWire.from.portIndex}-${newId}:0`, from: drawingWire.from, to: { componentId: newId, portIndex: 0 }, path: [...drawingWire.path, junction.point],
        };
        const newWireFromSplit1 = {
            id: `${intersectedWire.from.componentId}:${intersectedWire.from.portIndex}-${newId}:0`, from: intersectedWire.from, to: { componentId: newId, portIndex: 0 }, path: [...intersectedWire.path.slice(0, junction.segmentIndex + 1), junction.point],
        };
        const newWireFromSplit2 = {
            id: `${newId}:0-${intersectedWire.to.componentId}:${intersectedWire.to.portIndex}`, from: { componentId: newId, portIndex: 0 }, to: intersectedWire.to, path: [junction.point, ...intersectedWire.path.slice(junction.segmentIndex + 1)],
        };
        const newComponents = [...components, newNodeComponent];
        const newWires = [...wires.filter(w => w.id !== junction.wireId), newWireFromDrawing, newWireFromSplit1, newWireFromSplit2];
        const newNextId = newId + 1;
        this.setState({
            components: newComponents,
            wires: newWires,
            nextId: newNextId,
            drawingWire: null,
            junctionPreview: null,
        }, () => this.pushToHistory({ components: newComponents, wires: newWires, nextId: newNextId }));
    }
    completeWire(componentId, portIndex) {
        const { drawingWire, wires, components, nextId } = this.state;
        if (!drawingWire)
            return;
        if (drawingWire.from.componentId === componentId) {
            this.setState({ drawingWire: null });
            return;
        }
        const endPos = this.getPortPosition(componentId, portIndex);
        const newPath = [...drawingWire.path, endPos];
        const newWire = {
            id: `${drawingWire.from.componentId}:${drawingWire.from.portIndex}-${componentId}:${portIndex}`, from: drawingWire.from, to: { componentId, portIndex }, path: newPath,
        };
        const existingWire = wires.find(w => w.id === newWire.id || w.id === `${newWire.to.componentId}:${newWire.to.portIndex}-${newWire.from.componentId}:${newWire.from.portIndex}`);
        if (!existingWire) {
            const newWires = [...wires, newWire];
            this.setState({ wires: newWires, drawingWire: null }, () => this.pushToHistory({ components, wires: newWires, nextId }));
        } else {
            this.setState({ drawingWire: null });
        }
    }
    handleMouseMove(e) {
        const mousePos = this.getMousePosition(e);
        const { isDraggingMinimap, dragging, drawingWire, draggingHandle, wires } = this.state;
        if (isDraggingMinimap && this.minimapDragStartRef) {
            const { mouseX: startMouseX, mouseY: startMouseY, viewX: startViewX, viewY: startViewY } = this.minimapDragStartRef;
            const minimapSvg = this.minimapRef.current;
            if (!minimapSvg)
                return;
            const minimapRect = minimapSvg.getBoundingClientRect();
            const circuitBounds = this.getCircuitBounds();
            const scaleX = circuitBounds.width / minimapRect.width;
            const scaleY = circuitBounds.height / minimapRect.height;
            const dx = e.clientX - startMouseX;
            const dy = e.clientY - startMouseY;
            const newViewX = startViewX + dx * scaleX;
            const newViewY = startViewY + dy * scaleY;
            this.setState(prev => ({ viewBox: { ...prev.viewBox, x: newViewX, y: newViewY } }));
            return;
        }
        if (dragging) {
            const newX = snapToGrid(mousePos.x - dragging.offsetX);
            const newY = snapToGrid(mousePos.y - dragging.offsetY);
            this.setState(prev => ({
                components: prev.components.map(c => c.id === dragging.id ? { ...c, x: newX, y: newY } : c),
            }));
        }
        if (drawingWire) {
            this.isDraggingWire = true;
            let foundJunction = null;
            const snappedMousePos = { x: snapToGrid(mousePos.x), y: snapToGrid(mousePos.y) };
            for (const wire of wires) {
                for (let i = 0; i < wire.path.length - 1; i++) {
                    const p1 = wire.path[i];
                    const p2 = wire.path[i + 1];
                    const distSq = distToSegmentSquared(snappedMousePos, p1, p2);
                    if (distSq < 1) {
                        foundJunction = { point: snappedMousePos, wireId: wire.id, segmentIndex: i };
                        break;
                    }
                }
                if (foundJunction)
                    break;
            }
            const currentPos = foundJunction ? foundJunction.point : mousePos;
            this.setState({ junctionPreview: foundJunction, drawingWire: { ...drawingWire, currentPos } });
        }
        if (draggingHandle) {
            const snappedPos = { x: snapToGrid(mousePos.x), y: snapToGrid(mousePos.y) };
            this.setState(prev => ({
                wires: prev.wires.map(w => {
                    if (w.id === draggingHandle.wireId) {
                        const newPath = [...w.path];
                        newPath[draggingHandle.pointIndex] = snappedPos;
                        return { ...w, path: newPath };
                    }
                    return w;
                }),
            }));
        }
    }
    handleMouseUp() {
        if (this.state.dragging || this.state.draggingHandle) {
            this.pushToHistory({ components: this.state.components, wires: this.state.wires, nextId: this.state.nextId });
        }
        this.setState({ dragging: null, draggingHandle: null, isDraggingMinimap: false });
        this.minimapDragStartRef = null;
    }
    handleMouseDownOnComponent(e, id) {
        e.stopPropagation();
        const component = this.state.components.find(c => c.id === id);
        if (!component)
            return;
        this.setState({ selectedComponentId: id, selectedWireId: null, editingComponentId: null });
        const mousePos = this.getMousePosition(e);
        const offsetX = mousePos.x - component.x;
        const offsetY = mousePos.y - component.y;
        this.setState({ dragging: { id, offsetX, offsetY } });
    }
    handleDoubleClickOnComponent(id) {
        this.setState({ selectedComponentId: id, editingComponentId: id, selectedWireId: null });
    }
    handleMouseDownOnPort(e, componentId, portIndex) {
        e.stopPropagation();
        if (this.state.drawingWire) {
            this.completeWire(componentId, portIndex);
        } else {
            const pos = this.getPortPosition(componentId, portIndex);
            this.setState({
                drawingWire: { from: { componentId, portIndex }, path: [pos], currentPos: pos },
                selectedWireId: null,
                selectedComponentId: null,
            });
            this.isDraggingWire = false;
        }
    }
    handleMouseUpOnPort(e, componentId, portIndex) {
        e.stopPropagation();
        if (this.state.drawingWire) {
            if (this.state.drawingWire.from.componentId === componentId && !this.isDraggingWire) {
                return;
            }
            this.completeWire(componentId, portIndex);
        }
    }
    handleMouseDownOnHandle(e, wireId, pointIndex) {
        e.stopPropagation();
        this.setState({ draggingHandle: { wireId, pointIndex } });
    }
    handleDeleteHandle(e, wireId, pointIndex) {
        e.stopPropagation();
        e.preventDefault();
        const newWires = this.state.wires.map(w => {
            if (w.id === wireId && w.path.length > 2) {
                return { ...w, path: w.path.filter((_, idx) => idx !== pointIndex) };
            }
            return w;
        });
        this.setState({ wires: newWires }, () => this.pushToHistory({ components: this.state.components, wires: newWires, nextId: this.state.nextId }));
    }
    handleCanvasClick(e) {
        if (this.state.drawingWire && this.state.junctionPreview) {
            e.stopPropagation();
            this.createJunction(this.state.junctionPreview);
            return;
        }
        if (this.state.drawingWire) {
            e.stopPropagation();
            const mousePos = this.getMousePosition(e);
            const snappedPos = { x: snapToGrid(mousePos.x), y: snapToGrid(mousePos.y) };
            this.setState(prev => ({ drawingWire: { ...prev.drawingWire, path: [...prev.drawingWire.path, snappedPos] } }));
        } else {
            this.setState({ selectedWireId: null, selectedComponentId: null, editingComponentId: null });
        }
    }
    handleWireClick(e, wireId) {
        if (this.state.drawingWire) {
            return;
        }
        e.stopPropagation();
        this.setState({ selectedWireId: wireId, selectedComponentId: null });
    }
    handleDoubleClickOnWire(e, wireId) {
        e.stopPropagation();
        const mousePos = this.getMousePosition(e);
        const wireToUpdate = this.state.wires.find(w => w.id === wireId);
        if (!wireToUpdate)
            return;
        let closestSegmentIndex = -1;
        let minDistanceSq = Infinity;
        for (let i = 0; i < wireToUpdate.path.length - 1; i++) {
            const distSq = distToSegmentSquared(mousePos, wireToUpdate.path[i], wireToUpdate.path[i + 1]);
            if (distSq < minDistanceSq) {
                minDistanceSq = distSq;
                closestSegmentIndex = i;
            }
        }
        if (closestSegmentIndex !== -1) {
            const p1 = wireToUpdate.path[closestSegmentIndex];
            const p2 = wireToUpdate.path[closestSegmentIndex + 1];
            const closestPointOnSegment = getClosestPointOnSegment(mousePos, p1, p2);
            const newPoint = { x: snapToGrid(closestPointOnSegment.x), y: snapToGrid(closestPointOnSegment.y) };
            const newPath = [...wireToUpdate.path];
            newPath.splice(closestSegmentIndex + 1, 0, newPoint);
            const newWires = this.state.wires.map(w => w.id === wireId ? { ...w, path: newPath } : w);
            this.setState({ wires: newWires, selectedWireId: wireId }, () => this.pushToHistory({ components: this.state.components, wires: newWires, nextId: this.state.nextId }));
        }
    }
    handleDrop(e) {
        this.setState({ isDraggingNewComponent: false });
        e.preventDefault();
        const type = e.dataTransfer.getData('componentType');
        const COMPONENT_CONFIG = { ...this.state.baseComponentConfigs, ...this.state.customComponentConfigs };
        if (!type || !COMPONENT_CONFIG[type])
            return;
        const config = COMPONENT_CONFIG[type];
        const mousePos = this.getMousePosition(e);
        const x = snapToGrid(mousePos.x - config.width / 2);
        const y = snapToGrid(mousePos.y - config.height / 2);
        const componentCount = this.state.components.filter(c => c.type === type).length;
        const newComponentNumber = componentCount + 1;
        const newId = this.state.nextId;
        const newComponent = {
            id: newId,
            type,
            x,
            y,
            rotation: 0,
            title: type === 'text' ? 'Votre texte ici' : `${config.label} ${newComponentNumber}`,
            ...(type === 'text' && { width: config.width, height: config.height }),
        };
        const newComponents = [...this.state.components, newComponent];
        const newNextId = newId + 1;
        this.setState({ components: newComponents, nextId: newNextId }, () => this.pushToHistory({ components: newComponents, wires: this.state.wires, nextId: newNextId }));
    }
    handleDragStart(e, type) {
        e.dataTransfer.setData('componentType', type);
        this.setState({ isDraggingNewComponent: true });
    }
    handleDragEnd() {
        this.setState({ isDraggingNewComponent: false });
    }
    handleKeyDown(e) {
        if (e.ctrlKey || e.metaKey) {
            if (e.key === 'z') {
                e.preventDefault();
                this.handleUndo();
                return;
            }
            if (e.key === 'y') {
                e.preventDefault();
                this.handleRedo();
                return;
            }
        }
        if (e.key === 'Escape') {
            this.setState({ drawingWire: null, selectedWireId: null, selectedComponentId: null, editingComponentId: null });
        }
        if (e.key === 'Delete' || e.key === 'Backspace') {
            if (this.state.selectedComponentId !== null) {
                this.deleteComponent(this.state.selectedComponentId);
            }
            if (this.state.selectedWireId !== null) {
                this.deleteWire(this.state.selectedWireId);
            }
        }
    }
    handleWheel(e) {
        // Per user request, mouse wheel zooming has been disabled to prevent
        // interference on trackpads (e.g., on Chromebooks). Navigation
        // can be done by panning the view via the minimap.
        // We call preventDefault to stop the browser's default scroll action.
        e.preventDefault();
    }
    deleteComponent(componentId) {
        const { wires, components, nextId } = this.state;
        const wiresToRemove = wires.filter(w => w.from.componentId === componentId || w.to.componentId === componentId);
        const remainingWires = wires.filter(w => w.from.componentId !== componentId && w.to.componentId !== componentId);
        const remainingComponents = components.filter(c => c.id !== componentId);
        const nodesToProcess = new Set();
        wiresToRemove.forEach(w => {
            const otherEndComponentId = w.from.componentId === componentId ? w.to.componentId : w.from.componentId;
            const otherEndComp = remainingComponents.find(c => c.id === otherEndComponentId);
            if (otherEndComp?.type === 'node') {
                nodesToProcess.add(otherEndComp.id);
            }
        });
        const { newWires, newComponents } = simplifyNodes(remainingWires, remainingComponents, nodesToProcess);
        this.setState({
            wires: newWires,
            components: newComponents,
            selectedComponentId: this.state.selectedComponentId === componentId ? null : this.state.selectedComponentId
        }, () => this.pushToHistory({ components: newComponents, wires: newWires, nextId }));
    }
    deleteWire(wireId) {
        const { wires, components, nextId } = this.state;
        const wireToDelete = wires.find(w => w.id === wireId);
        if (!wireToDelete)
            return;
        const remainingWires = wires.filter(w => w.id !== wireId);
        const nodesToProcess = new Set();
        const fromComp = components.find(c => c.id === wireToDelete.from.componentId);
        if (fromComp?.type === 'node')
            nodesToProcess.add(fromComp.id);
        const toComp = components.find(c => c.id === wireToDelete.to.componentId);
        if (toComp?.type === 'node')
            nodesToProcess.add(toComp.id);
        const { newWires, newComponents } = simplifyNodes(remainingWires, components, nodesToProcess);
        this.setState({
            wires: newWires,
            components: newComponents,
            selectedWireId: this.state.selectedWireId === wireId ? null : this.state.selectedWireId
        }, () => this.pushToHistory({ components: newComponents, wires: newWires, nextId }));
    }
    handleRotateComponent(componentId) {
        const newComponents = this.state.components.map(c => c.id === componentId ? { ...c, rotation: (c.rotation + 90) % 360 } : c);
        this.setState({ components: newComponents }, () => this.pushToHistory({ components: newComponents, wires: this.state.wires, nextId: this.state.nextId }));
    }
    handleUpdateComponent(componentId, updates) {
        this.setState(prev => ({
            components: prev.components.map(c => c.id === componentId ? { ...c, ...updates } : c)
        }));
    }
    handleComponentUpdateBlur() {
        this.pushToHistory({ components: this.state.components, wires: this.state.wires, nextId: this.state.nextId });
        this.setState({ editingComponentId: null });
    }
    handleSave() {
        const { components, wires, baseComponentConfigs, customComponentConfigs } = this.state;
        const COMPONENT_CONFIG = { ...baseComponentConfigs, ...customComponentConfigs };
        const data = JSON.stringify({
            componentLibrary: COMPONENT_CONFIG,
            circuit: { components, wires }
        }, null, 2);
        const blob = new Blob([data], { type: 'application/json' });
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = 'circuit.json';
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        URL.revokeObjectURL(url);
    }
    handleLoadClick() {
        this.fileInputRef.current?.click();
    }
    async generateCircuitCanvas() {
        const { components, wires, baseComponentConfigs, customComponentConfigs } = this.state;
        const COMPONENT_CONFIG = { ...baseComponentConfigs, ...customComponentConfigs };
        if (components.length === 0 && wires.length === 0) {
            return null;
        }
        const padding = 50;
        let minX = Infinity, minY = Infinity, maxX = -Infinity, maxY = -Infinity;
        components.forEach(comp => {
            const config = COMPONENT_CONFIG[comp.type];
            if (!config)
                return;
            const width = comp.width ?? config.width;
            const height = comp.height ?? config.height;
            const cx = width / 2;
            const cy = height / 2;
            const rad = (comp.rotation || 0) * Math.PI / 180;
            const cos = Math.cos(rad);
            const sin = Math.sin(rad);
            const corners = [
                { x: 0, y: 0 }, { x: width, y: 0 }, { x: width, y: height }, { x: 0, y: height }
            ];
            corners.forEach(corner => {
                const translatedX = corner.x - cx;
                const translatedY = corner.y - cy;
                const rotatedX = translatedX * cos - translatedY * sin;
                const rotatedY = translatedX * sin + translatedY * cos;
                const finalX = comp.x + rotatedX + cx;
                const finalY = comp.y + rotatedY + cy;
                minX = Math.min(minX, finalX);
                minY = Math.min(minY, finalY);
                maxX = Math.max(maxX, finalX);
                maxY = Math.max(maxY, finalY);
            });
            if (comp.type !== 'node' && comp.title) {
                const { foProps } = this.getTitleProps(comp, COMPONENT_CONFIG);
                const titleX = comp.x + foProps.x;
                const titleY = comp.y + foProps.y;
                minX = Math.min(minX, titleX);
                minY = Math.min(minY, titleY);
                maxX = Math.max(maxX, titleX + foProps.width);
                maxY = Math.max(maxY, titleY + foProps.height);
            }
        });
        wires.forEach(wire => {
            wire.path.forEach(p => {
                minX = Math.min(minX, p.x);
                minY = Math.min(minY, p.y);
                maxX = Math.max(maxX, p.x);
                maxY = Math.max(maxY, p.y);
            });
        });
        if (!isFinite(minX)) {
            return null;
        }
        const contentWidth = maxX - minX;
        const contentHeight = maxY - minY;
        const svgWidth = contentWidth + padding * 2;
        const svgHeight = contentHeight + padding * 2;
        const viewBox = `${minX - padding} ${minY - padding} ${svgWidth} ${svgHeight}`;
        const tempContainer = document.createElement('div');
        tempContainer.style.position = 'absolute';
        tempContainer.style.left = '-9999px';
        document.body.appendChild(tempContainer);
        const SvgForExport = (React.createElement("svg", { xmlns: "http://www.w3.org/2000/svg", width: svgWidth, height: svgHeight, viewBox: viewBox }, React.createElement("rect", { x: minX - padding, y: minY - padding, width: svgWidth, height: svgHeight, fill: "white" }), wires.map(wire => (React.createElement(WireRenderer, { key: wire.id, wire: wire, isSelected: false, onClick: () => { }, onDoubleClick: () => { }, onDelete: () => { }, onHandleMouseDown: () => { }, onHandleContextMenu: () => { } }))), components.map(comp => {
            const config = COMPONENT_CONFIG[comp.type];
            if (!config)
                return null;
            return (React.createElement(ComponentRenderer, { key: comp.id, comp: comp, config: config, isSelected: false, isEditing: false, isComponentConnected: this.isComponentConnected(comp.id), isExporting: true, onMouseDown: () => { }, onUpdateComponent: () => { }, onTitleBlur: () => { }, onDoubleClick: () => { }, onRotate: () => { }, onDelete: () => { }, onPortMouseDown: () => { }, onPortMouseUp: () => { }, getTitleProps: (c) => this.getTitleProps(c, COMPONENT_CONFIG) }));
        })));
        const root = ReactDOM.createRoot(tempContainer);
        root.render(SvgForExport);
        return new Promise((resolve, reject) => {
            setTimeout(() => {
                try {
                    const svgEl = tempContainer.querySelector('svg');
                    if (!svgEl)
                        throw new Error("Élément SVG pour l'exportation introuvable.");
                    const svgString = new XMLSerializer().serializeToString(svgEl);
                    const blob = new Blob([svgString], { type: 'image/svg+xml;charset=utf-8' });
                    const url = URL.createObjectURL(blob);
                    const img = new Image();
                    img.onload = () => {
                        const canvas = document.createElement('canvas');
                        canvas.width = svgWidth;
                        canvas.height = svgHeight;
                        const ctx = canvas.getContext('2d');
                        if (ctx) {
                            ctx.drawImage(img, 0, 0);
                            resolve(canvas);
                        } else {
                            reject(new Error("Impossible d'obtenir le contexte 2D du canevas."));
                        }
                        URL.revokeObjectURL(url);
                    };
                    img.onerror = (err) => {
                        URL.revokeObjectURL(url);
                        reject(err);
                    };
                    img.src = url;
                } catch (error) {
                    reject(error);
                } finally {
                    root.unmount();
                    document.body.removeChild(tempContainer);
                }
            }, 100);
        });
    }
    async handleGenerateImage() {
        if (this.state.components.length === 0 && this.state.wires.length === 0) {
            alert("Le circuit est vide.");
            return;
        }
        try {
            const canvas = await this.generateCircuitCanvas();
            if (canvas) {
                const pngUrl = canvas.toDataURL('image/png');
                const a = document.createElement('a');
                a.href = pngUrl;
                a.download = 'circuit.png';
                document.body.appendChild(a);
                a.click();
                document.body.removeChild(a);
            } else if (isFinite(this.state.components.reduce((acc, c) => acc + c.x, 0))) {
                alert("Impossible de générer l'image du circuit.");
            }
        } catch (err) {
            console.error("Erreur lors de la génération de l'image : ", err);
            alert(`Une erreur est survenue lors de la génération de l'image : ${err.message}`);
        }
    }
    async handleCopyImage() {
        if (this.state.components.length === 0 && this.state.wires.length === 0) {
            alert("Le circuit est vide.");
            return;
        }
        try {
            const canvas = await this.generateCircuitCanvas();
            if (canvas) {
                canvas.toBlob(async (blob) => {
                    if (blob) {
                        try {
                            if (!navigator.clipboard?.write) {
                                throw new Error("La copie dans le presse-papiers n'est pas prise en charge par votre navigateur ou le contexte n'est pas sécurisé (HTTPS).");
                            }
                            await navigator.clipboard.write([
                                new ClipboardItem({ 'image/png': blob })
                            ]);
                            alert('Image copiée dans le presse-papiers !');
                        } catch (copyError) {
                            console.error("Échec de la copie de l'image : ", copyError);
                            alert(`La copie de l'image a échoué : ${copyError.message}`);
                        }
                    } else {
                        throw new Error("Impossible de créer un blob à partir du canevas.");
                    }
                }, 'image/png');
            } else if (isFinite(this.state.components.reduce((acc, c) => acc + c.x, 0))) {
                alert("Impossible de préparer l'image pour la copie.");
            }
        } catch (err) {
            console.error("Erreur lors de la copie de l'image : ", err);
            alert(`Une erreur est survenue lors de la préparation de l'image : ${err.message}`);
        }
    }
    handleFileChange(e) {
        const file = e.target.files?.[0];
        if (!file)
            return;
        const reader = new FileReader();
        reader.onload = (event) => {
            try {
                const result = event.target?.result;
                if (typeof result !== 'string') {
                    alert("Erreur lors de la lecture du fichier de circuit.");
                    return;
                }
                const data = JSON.parse(result);
                const isNewFormat = data && data.circuit && data.componentLibrary;
                const circuitData = isNewFormat ? data.circuit : data;
                const libraryData = isNewFormat ? data.componentLibrary : null;
                if (!circuitData || !Array.isArray(circuitData.components) || !Array.isArray(circuitData.wires)) {
                    alert('Format de fichier de circuit invalide.');
                    return;
                }
                const validatedWires = circuitData.wires.filter((w) => w && typeof w === 'object' && w.from && w.to && Array.isArray(w.path));
                if (validatedWires.length !== circuitData.wires.length) {
                    console.warn("Certains fils du fichier chargé ont été ignorés en raison d'un format non valide.");
                }
                let allConfigsForPreview = { ...this.state.baseComponentConfigs, ...this.state.customComponentConfigs };
                if (libraryData) {
                    for (const typeKey in libraryData) {
                        if (!allConfigsForPreview[typeKey]) {
                            const validConfig = normalizeAndValidateConfig(libraryData[typeKey]);
                            if (validConfig) {
                                allConfigsForPreview[typeKey] = validConfig;
                            }
                        }
                    }
                }
                const validComponents = circuitData.components.filter((c) => c && c.type && allConfigsForPreview[c.type]);
                if (validComponents.length !== circuitData.components.length) {
                    console.warn("Certains composants du fichier chargé ont été ignorés car leur type est inconnu ou leur format est non valide.");
                }
                this.setState({
                    isPreviewModalOpen: true,
                    previewData: {
                        components: validComponents,
                        wires: validatedWires,
                        libraryData,
                        allConfigsForPreview,
                    }
                });
            } catch (error) {
                alert('Erreur lors de la lecture du fichier de circuit.');
                console.error(error);
            }
        };
        reader.readAsText(file);
        if (e.target)
            e.target.value = '';
    }
    handleConfirmLoad() {
        const { previewData } = this.state;
        if (!previewData)
            return;
        const { components, wires, libraryData, allConfigsForPreview } = previewData;
        let updatedCustomConfigs = { ...this.state.customComponentConfigs };
        let configsAdded = false;
        if (libraryData) {
            const currentFullConfig = { ...this.state.baseComponentConfigs, ...this.state.customComponentConfigs };
            for (const typeKey in libraryData) {
                if (!currentFullConfig[typeKey]) {
                    const validConfig = normalizeAndValidateConfig(libraryData[typeKey]);
                    if (validConfig) {
                        updatedCustomConfigs[typeKey] = validConfig;
                        configsAdded = true;
                    }
                }
            }
        }
        if (configsAdded) {
            this.setState({ customComponentConfigs: updatedCustomConfigs });
            localStorage.setItem('customComponents', JSON.stringify(updatedCustomConfigs));
        }
        let maxId = 0;
        const loadedComponentsWithDefaults = components.map((c) => {
            maxId = Math.max(maxId, c.id);
            const config = allConfigsForPreview[c.type];
            const defaultTitle = config ? `${config.label || 'Composant'} ${c.id}` : `Composant ${c.id}`;
            return { ...c, rotation: c.rotation || 0, title: c.title || defaultTitle };
        });
        const newNextId = maxId + 1;
        const newState = { components: loadedComponentsWithDefaults, wires: wires, nextId: newNextId };
        this.setState({
            ...newState,
            isPreviewModalOpen: false,
            previewData: null,
            selectedComponentId: null,
            selectedWireId: null,
        }, () => this.resetHistory(newState));
    }
    handleCancelLoad() {
        this.setState({ isPreviewModalOpen: false, previewData: null });
    }
    handleReset() {
        const newComponents = [];
        const newWires = [];
        const newNextId = 1;
        const newState = {
            components: newComponents,
            wires: newWires,
            nextId: newNextId,
            dragging: null,
            drawingWire: null,
            draggingHandle: null,
            junctionPreview: null,
            selectedComponentId: null,
            selectedWireId: null,
        };
        this.setState(newState, () => this.resetHistory({ components: newComponents, wires: newWires, nextId: newNextId }));
    }
    handleMinimapDragStart(e) {
        e.stopPropagation();
        this.setState({ isDraggingMinimap: true });
        this.minimapDragStartRef = {
            mouseX: e.clientX,
            mouseY: e.clientY,
            viewX: this.state.viewBox.x,
            viewY: this.state.viewBox.y,
        };
    }
    renderPreviewModal() {
        const { previewData } = this.state;
        if (!previewData)
            return null;
        const { components, wires, allConfigsForPreview } = previewData;
        const padding = 50;
        let minX = Infinity, minY = Infinity, maxX = -Infinity, maxY = -Infinity;
        if (components.length === 0 && wires.length === 0) {
            minX = 0;
            minY = 0;
            maxX = 400;
            maxY = 300;
        } else {
            components.forEach(comp => {
                const config = allConfigsForPreview[comp.type];
                if (!config)
                    return;
                const width = comp.width ?? config.width;
                const height = comp.height ?? config.height;
                minX = Math.min(minX, comp.x);
                minY = Math.min(minY, comp.y);
                maxX = Math.max(maxX, comp.x + width);
                maxY = Math.max(maxY, comp.y + height);
            });
            wires.forEach(wire => {
                wire.path.forEach(p => {
                    minX = Math.min(minX, p.x);
                    minY = Math.min(minY, p.y);
                    maxX = Math.max(maxX, p.x);
                    maxY = Math.max(maxY, p.y);
                });
            });
        }
        if (!isFinite(minX)) { // Handle case with wires but no components
            minX = 0;
            minY = 0;
            maxX = 400;
            maxY = 300;
        }
        const contentWidth = maxX - minX;
        const contentHeight = maxY - minY;
        const svgWidth = contentWidth + padding * 2;
        const svgHeight = contentHeight + padding * 2;
        const viewBox = `${minX - padding} ${minY - padding} ${svgWidth} ${svgHeight}`;
        return (React.createElement("div", { className: "position-absolute top-0 start-0 w-100 h-100 d-flex align-items-center justify-content-center z-3", style: { backgroundColor: 'rgba(0,0,0,0.7)' }, onClick: this.handleCancelLoad }, React.createElement("div", { className: "bg-white rounded-3 shadow-lg p-4 w-100 d-flex flex-column", style: { maxWidth: '42rem', maxHeight: '90vh' }, onClick: e => e.stopPropagation() }, React.createElement("h2", { className: "h2 fw-bold text-primary mb-4" }, "Aperçu du Circuit"), React.createElement("div", { className: "flex-grow-1 bg-light rounded border overflow-hidden mb-4" }, React.createElement("svg", { xmlns: "http://www.w3.org/2000/svg", width: "100%", height: "100%", viewBox: viewBox, className: "bg-white" }, React.createElement("defs", null, React.createElement("pattern", { id: "preview-grid", width: GRID_SIZE, height: GRID_SIZE, patternUnits: "userSpaceOnUse" }, React.createElement("circle", { cx: "1", cy: "1", r: "1", fill: "#e2e8f0" }))), React.createElement("rect", { x: minX - padding, y: minY - padding, width: svgWidth, height: svgHeight, fill: "url(#preview-grid)" }), wires.map(wire => (React.createElement(WireRenderer, { key: wire.id, wire: wire, isSelected: false, onClick: () => { }, onDoubleClick: () => { }, onDelete: () => { }, onHandleMouseDown: () => { }, onHandleContextMenu: () => { } }))), components.map(comp => {
            const config = allConfigsForPreview[comp.type];
            if (!config)
                return null;
            return (React.createElement(ComponentRenderer, { key: comp.id, comp: comp, config: config, isSelected: false, isEditing: false, isComponentConnected: false, isExporting: true, onMouseDown: () => { }, onUpdateComponent: () => { }, onTitleBlur: () => { }, onDoubleClick: () => { }, onRotate: () => { }, onDelete: () => { }, onPortMouseDown: () => { }, onPortMouseUp: () => { }, getTitleProps: (c) => this.getTitleProps(c, allConfigsForPreview) }));
        }))), React.createElement("div", { className: "text-center text-muted mb-4" }, React.createElement("p", null, `${components.length} composant(s) et ${wires.length} fil(s) seront chargés.`)), React.createElement("div", { className: "mt-auto d-flex justify-content-end gap-3" }, React.createElement("button", { onClick: this.handleCancelLoad, className: "btn btn-secondary fw-bold" }, "Annuler"), React.createElement("button", { onClick: this.handleConfirmLoad, className: "btn btn-success fw-bold" }, "Charger le Circuit")))));
    }
    render() {
        const { components, wires, drawingWire, selectedWireId, selectedComponentId, editingComponentId, junctionPreview, baseComponentConfigs, customComponentConfigs, isModalOpen, viewBox, history, historyIndex } = this.state;
        const COMPONENT_CONFIG = { ...baseComponentConfigs, ...customComponentConfigs };
        const canUndo = historyIndex > 0;
        const canRedo = historyIndex < history.length - 1;
        const circuitBounds = this.getCircuitBounds();
        return (React.createElement("div", { className: "vh-100 d-flex flex-column bg-white text-dark" }, React.createElement("header", { className: "d-flex flex-shrink-0 align-items-center p-2 bg-light border-bottom shadow-sm z-2" }, React.createElement("img", { src: "logo.png", alt: "Logo", style: { height: '40px', width: '40px' }, className: "me-3" }), React.createElement("h1", { className: "h2 fw-bold text-secondary" }, "Éditeur de diagramme schématique")), React.createElement("div", { className: "d-flex flex-grow-1 overflow-hidden position-relative" }, React.createElement("input", { type: "file", ref: this.fileInputRef, onChange: this.handleFileChange, accept: ".json", className: "d-none" }), React.createElement("input", { type: "file", ref: this.componentsFileInputRef, onChange: (e) => {
            const file = e.target.files?.[0];
            if (!file)
                return;
            const reader = new FileReader();
            reader.onload = (event) => {
                try {
                    const result = event.target?.result;
                    if (typeof result !== 'string') {
                        alert("Erreur: Le contenu du fichier n'a pas pu être lu comme du texte.");
                        return;
                    }
                    const loadedConfigs = JSON.parse(result);
                    if (typeof loadedConfigs !== 'object' || loadedConfigs === null || Array.isArray(loadedConfigs)) {
                        throw new Error("Le fichier doit contenir un objet JSON de configurations de composants.");
                    }
                    const validNewConfigs = {};
                    let importedCount = 0;
                    let invalidCount = 0;
                    for (const key in loadedConfigs) {
                        const validConfig = normalizeAndValidateConfig(loadedConfigs[key]);
                        if (validConfig) {
                            validNewConfigs[key] = validConfig;
                            importedCount++;
                        } else {
                            invalidCount++;
                        }
                    }
                    if (importedCount > 0) {
                        const mergedConfigs = { ...this.state.customComponentConfigs, ...validNewConfigs };
                        this.setState({ customComponentConfigs: mergedConfigs });
                        localStorage.setItem('customComponents', JSON.stringify(mergedConfigs));
                        alert(`${importedCount} composant(s) importé(s).${invalidCount > 0 ? ` ${invalidCount} invalide(s) ignoré(s).` : ''}`);
                    } else {
                        alert(`Aucun composant valide trouvé.${invalidCount > 0 ? ` ${invalidCount} invalide(s) ignoré(s).` : ''}`);
                    }
                } catch (error) {
                    alert(`Erreur: ${error.message}`);
                }
            };
            reader.readAsText(file);
            if (e.target)
                e.target.value = '';
        }, accept: ".json", className: "d-none" }), React.createElement(Sidebar, {
            componentConfig: COMPONENT_CONFIG,
            onDragStart: this.handleDragStart,
            onSave: this.handleSave,
            onLoad: this.handleLoadClick,
            onReset: this.handleReset,
            onManageComponents: () => this.setState({ isModalOpen: true }),
            onUndo: this.handleUndo,
            onRedo: this.handleRedo,
            canUndo: canUndo,
            canRedo: canRedo,
            onGenerateImage: this.handleGenerateImage,
            onCopyImage: this.handleCopyImage
        }), React.createElement("main", { className: "flex-grow-1 position-relative", onMouseMove: this.handleMouseMove, onMouseUp: this.handleMouseUp, onMouseLeave: this.handleMouseUp, onDragEnd: this.handleDragEnd }, React.createElement("svg", {
            ref: this.canvasRef,
            className: "w-100 h-100 bg-white",
            onDrop: this.handleDrop,
            onDragOver: (e) => e.preventDefault(),
            onClick: this.handleCanvasClick,
            onWheel: this.handleWheel,
            viewBox: `${viewBox.x} ${viewBox.y} ${viewBox.width} ${viewBox.height}`
        }, React.createElement("defs", null, React.createElement("pattern", { id: "grid", width: GRID_SIZE, height: GRID_SIZE, patternUnits: "userSpaceOnUse" }, React.createElement("circle", { cx: "1", cy: "1", r: "1", fill: "#cbd5e1" }))), React.createElement("rect", { x: viewBox.x, y: viewBox.y, width: viewBox.width, height: viewBox.height, fill: "url(#grid)" }), wires.map(wire => (React.createElement(WireRenderer, {
            key: wire.id,
            wire: wire,
            isSelected: selectedWireId === wire.id,
            onClick: this.handleWireClick,
            onDoubleClick: this.handleDoubleClickOnWire,
            onDelete: this.deleteWire,
            onHandleMouseDown: this.handleMouseDownOnHandle,
            onHandleContextMenu: this.handleDeleteHandle
        }))), drawingWire && (React.createElement("path", {
            d: generateOrthogonalPathString([...drawingWire.path, drawingWire.currentPos]),
            stroke: "black",
            strokeWidth: "2",
            strokeDasharray: "5,5",
            fill: "none",
            className: "pointer-events-none"
        })), junctionPreview && (React.createElement("circle", { cx: junctionPreview.point.x, cy: junctionPreview.point.y, r: "6", fill: "black", className: "pointer-events-none" })), components.map(comp => {
            const config = COMPONENT_CONFIG[comp.type];
            if (!config)
                return null;
            return (React.createElement(ComponentRenderer, {
                key: comp.id,
                comp: comp,
                config: config,
                isSelected: selectedComponentId === comp.id,
                isEditing: editingComponentId === comp.id,
                isComponentConnected: this.isComponentConnected(comp.id),
                drawingWire: drawingWire,
                onMouseDown: this.handleMouseDownOnComponent,
                onUpdateComponent: this.handleUpdateComponent,
                onTitleBlur: this.handleComponentUpdateBlur,
                onDoubleClick: this.handleDoubleClickOnComponent,
                onRotate: this.handleRotateComponent,
                onDelete: this.deleteComponent,
                onPortMouseDown: this.handleMouseDownOnPort,
                onPortMouseUp: this.handleMouseUpOnPort,
                getTitleProps: this.getTitleProps
            }));
        })), React.createElement(Minimap, {
            components: components,
            wires: wires,
            viewBox: viewBox,
            circuitBounds: circuitBounds,
            componentConfig: COMPONENT_CONFIG,
            onDragStart: this.handleMinimapDragStart,
            minimapRef: this.minimapRef
        })), React.createElement(ComponentManagerModal, {
            isOpen: isModalOpen,
            onClose: () => this.setState({ isModalOpen: false }),
            customComponentConfigs: customComponentConfigs,
            onUpdateCustomComponents: (newConfigs) => {
                this.setState({ customComponentConfigs: newConfigs });
                localStorage.setItem('customComponents', JSON.stringify(newConfigs));
            },
            allComponentConfigs: COMPONENT_CONFIG,
            onImportClick: () => this.componentsFileInputRef.current?.click()
        }), this.state.isPreviewModalOpen && this.renderPreviewModal())));
    }
}

const root = ReactDOM.createRoot(document.getElementById('root'));
root.render(React.createElement(React.StrictMode, null, React.createElement(App, null)));
