document.addEventListener('DOMContentLoaded', () => {
    // Configuration de Day.js
    dayjs.extend(window.dayjs_plugin_customParseFormat);
    dayjs.extend(window.dayjs_plugin_utc);
    dayjs.locale('fr');

    let competenciesData = null, workableDays = null, studentsData = null;
    let lastStudentSummaryHTML = '';
    let totalWeeks = 52; 
    let isSingleStudentMode = false;

    // Références DOM
    const modeSelectionContainer = document.getElementById('mode-selection-container');
    const selectCohortModeBtn = document.getElementById('select-cohort-mode');
    const selectSingleModeBtn = document.getElementById('select-single-mode');
    const controlsContainer = document.querySelector('.controls');
    const studentSelect = document.getElementById('student-select');
    const mainContent = document.getElementById('main-content');
    const summaryDiv = document.querySelector('.summary');
    const skillsTable = document.getElementById('skills-table');
    const ganttContainer = document.getElementById('gantt-chart-container');
    const ganttChart = document.querySelector('.gantt-chart');
    const scrollLeftBtn = document.getElementById('scroll-left');
    const scrollRightBtn = document.getElementById('scroll-right');
    const tabButtons = document.querySelectorAll(".tab-btn");
    const scheduleView = document.getElementById('schedule-view');
    const evaluationsView = document.getElementById('evaluations-view');
    const studentSelectionContainer = document.getElementById('student-selection-container');
    const loadingStatusContainer = document.getElementById('loading-status-container');
    const manualUploadContainer = document.getElementById('manual-upload-container');
    const matriceInput = document.getElementById('matrice-file');
    const calendrierInput = document.getElementById('calendrier-file');
    const elevesInput = document.getElementById('eleves-file');
    const monthSelector = document.getElementById('month-selector');
    const calendarHeader = document.getElementById('calendar-header');
    const localSelector = document.getElementById('local-selector');
    const occupationMonthSelector = document.getElementById('occupation-month-selector');
    const occupationCalendarGrid = document.getElementById('occupation-calendar-grid');
    const cohortTimelineView = document.getElementById('cohort-timeline-view');
    const stageView = document.getElementById('stage-view');
    const singleStudentGenerator = document.getElementById('single-student-generator');
    const studentNameInput = document.getElementById('student-name-input');
    const studentDateInput = document.getElementById('student-date-input');
    const generatePlanBtn = document.getElementById('generate-plan-btn');
    const scheduleTitle = document.querySelector('#schedule-title');

    const parseCsvData = (csvText) => {
        const lines = csvText.split('\n').filter(r => r.trim());
        if (lines.length === 0) return [];
        const header = lines[0];
        const commaCount = (header.match(/,/g) || []).length;
        const semicolonCount = (header.match(/;/g) || []).length;
        const delimiter = semicolonCount > commaCount ? ';' : ',';
        return lines.map(line => line.split(delimiter));
    };

    const parseFile = file => new Promise(resolve => {
        const reader = new FileReader();
        reader.onload = e => resolve(parseCsvData(e.target.result));
        reader.readAsText(file, 'UTF-8');
    });

    const setStatus = (id, message, type) => {
        const el = document.getElementById(id);
        if (el) { el.textContent = message; el.className = `file-status ${type}`; }
    };

    const loadCoreFiles = async () => {
        const [matriceRes, calendrierRes] = await Promise.all([ fetch('matrice.csv'), fetch('calendrier.csv') ]);
        if (!matriceRes.ok) throw new Error('matrice.csv manquant');
        if (!calendrierRes.ok) throw new Error('calendrier.csv manquant');
        
        const decoder = new TextDecoder('utf-8');
        const [matriceBuffer, calendrierBuffer] = await Promise.all([ matriceRes.arrayBuffer(), calendrierRes.arrayBuffer() ]);
        processCompetenciesData(parseCsvData(decoder.decode(matriceBuffer)));
        processCalendarData(parseCsvData(decoder.decode(calendrierBuffer)));
    };
    
    const startApplication = async () => {
        try {
            await loadCoreFiles();
            modeSelectionContainer.hidden = false;
        } catch (error) {
            console.error("Erreur critique au chargement des fichiers de base:", error);
            controlsContainer.hidden = false;
            manualUploadContainer.hidden = false;
            loadingStatusContainer.hidden = true;
            document.getElementById('eleves-file').parentElement.hidden = true;
        }
    };

    selectCohortModeBtn.addEventListener('click', async () => {
        isSingleStudentMode = false;
        modeSelectionContainer.hidden = true;
        controlsContainer.hidden = false;
        loadingStatusContainer.hidden = false;
        setStatus('matrice-status', '✔ Chargé', 'loaded');
        setStatus('calendrier-status', '✔ Chargé', 'loaded');
        
        try {
            const elevesRes = await fetch('eleves.csv');
            if (!elevesRes.ok) throw new Error("Fichier eleves.csv non trouvé sur le serveur.");
            const elevesBuffer = await elevesRes.arrayBuffer();
            processStudentsData(parseCsvData(new TextDecoder('utf-8').decode(elevesBuffer)));
            setStatus('eleves-status', '✔ Chargé', 'loaded');
            setupCohortMode();
        } catch (error) {
            console.error(error);
            setStatus('eleves-status', '✖ Non trouvé', 'error');
            singleStudentGenerator.hidden = true;
            studentSelectionContainer.hidden = true;
            manualUploadContainer.hidden = false;
            document.getElementById('matrice-file').parentElement.hidden = true;
            document.getElementById('calendrier-file').parentElement.hidden = true;
            document.getElementById('eleves-file').parentElement.hidden = false;
        }
    });

    selectSingleModeBtn.addEventListener('click', () => {
        isSingleStudentMode = true;
        modeSelectionContainer.hidden = true;
        controlsContainer.hidden = false;
        loadingStatusContainer.hidden = true;
        setupSingleStudentMode();
    });

    const setupCohortMode = () => {
        loadingStatusContainer.hidden = true;
        manualUploadContainer.hidden = true;
        singleStudentGenerator.hidden = true;
        studentSelectionContainer.hidden = false;
        
        document.querySelectorAll('.tab-btn').forEach(btn => btn.hidden = false);
        // MODIFICATION : L'onglet par défaut est 'suivi' pour afficher le sélecteur d'élèves
        document.querySelector('.tab-btn[data-tab="suivi"]').click();

        studentSelect.innerHTML = '<option value="">-- Sélectionnez un élève --</option>';
        studentsData.forEach(student => studentSelect.add(new Option(student.name, student.startDate)));
        
        populateMonthSelectors();
        populateLocalSelector();
        updateTabTitles('');
        updateEvaluationsTab();
        updateCohortTab(); 
        updateStageTab();
    };
    
    const setupSingleStudentMode = () => {
        studentSelectionContainer.hidden = true;
        singleStudentGenerator.hidden = false;
        manualUploadContainer.hidden = true;
        loadingStatusContainer.hidden = true;

        document.querySelectorAll('.tab-btn').forEach(btn => btn.hidden = false);
        document.querySelector('.tab-btn[data-tab="cohorte"]').hidden = true;
        document.querySelector('.tab-btn[data-tab="stage"]').hidden = true;
        document.querySelector('.tab-btn[data-tab="locaux"]').hidden = true;
        document.querySelector('.tab-btn[data-tab="evaluations"]').hidden = true;
        
        document.querySelector('.tab-btn[data-tab="suivi"]').click();
        
        populateMonthSelectors();
        updateTabTitles('');
    };

    const processCompetenciesData = (data) => {
        if (data.length < 2) {
            competenciesData = [];
            return;
        }
        const header = data[0];
        const fixedColumns = 6; 
        totalWeeks = header.length - fixedColumns;

        competenciesData = data.slice(1).map((row) => {
            const weeklyHours = row.slice(fixedColumns).map(h => parseInt(h, 10) || 0);
            return {
                name: row[2].trim(),
                shortName: `C-${(row[0]).toString().padStart(2, '0')}`,
                totalHours: parseInt(row[3], 10) || 0,
                type: row[4].trim(),
                local: row[5].trim(),
                weeklyHours: weeklyHours,
                startWeek: weeklyHours.findIndex(h => h > 0),
                endWeek: weeklyHours.findLastIndex(h => h > 0)
            };
        }).filter(c => c.totalHours > 0 && c.startWeek !== -1);
    };

    const processCalendarData = (data) => {
         workableDays = data.map(row => {
            if (row.length < 3) return null;
            const dateString = `${row[0].trim()},${row[1].trim().toLowerCase()},${row[2].trim()}`;
            const date = dayjs.utc(dateString, 'D,MMMM,YYYY', 'fr');
            return date.isValid() ? date : null;
        }).filter(Boolean).sort((a, b) => a.valueOf() - b.valueOf());
    };

    const processStudentsData = (data) => {
        studentsData = data.slice(1).map(row => ({ name: row[0].trim(), startDate: row[1].trim() }));
    };

    let manualFilesLoaded = { matrice: false, calendrier: false, eleves: false };
    const checkManualFiles = () => { if (manualFilesLoaded.matrice && manualFilesLoaded.calendrier && manualFilesLoaded.eleves) setupCohortMode(); };

    matriceInput.addEventListener('change', async (e) => { processCompetenciesData(await parseFile(e.target.files[0])); manualFilesLoaded.matrice = true; checkManualFiles(); });
    calendrierInput.addEventListener('change', async (e) => { processCalendarData(await parseFile(e.target.files[0])); manualFilesLoaded.calendrier = true; checkManualFiles(); });
    elevesInput.addEventListener('change', async (e) => { 
        processStudentsData(await parseFile(e.target.files[0])); 
        manualFilesLoaded.eleves = true;
        // MODIFICATION : Logique de chargement manuel corrigée
        if (competenciesData && workableDays) {
            setupCohortMode();
        } else {
            checkManualFiles();
        }
    });

    const populateMonthSelectors = () => {
        if (!workableDays || workableDays.length === 0) return;
        const uniqueMonths = new Set();
        workableDays.forEach(day => uniqueMonths.add(day.format('YYYY-MM')));
        const sortedMonths = Array.from(uniqueMonths).sort();
        
        monthSelector.innerHTML = '';
        if (occupationMonthSelector) occupationMonthSelector.innerHTML = '';
        
        sortedMonths.forEach(month => {
            const monthName = dayjs.utc(month).format('MMMM YYYY');
            const capitalizedMonthName = monthName.charAt(0).toUpperCase() + monthName.slice(1);
            monthSelector.add(new Option(capitalizedMonthName, month));
            if (occupationMonthSelector) occupationMonthSelector.add(new Option(capitalizedMonthName, month));
        });

        const currentMonth = dayjs.utc().format('YYYY-MM');
        if (sortedMonths.includes(currentMonth)) {
            monthSelector.value = currentMonth;
            if (occupationMonthSelector) occupationMonthSelector.value = currentMonth;
        }
    };

    const populateLocalSelector = () => {
        if (!competenciesData) return;
        const uniqueLocals = [...new Set(competenciesData.map(c => c.local))].sort();
        localSelector.innerHTML = '<option value="">-- Sélectionnez un local --</option>';
        uniqueLocals.forEach(local => {
            if (local) localSelector.add(new Option(local, local));
        });
    };

    const calculateProgress = (enrollmentDateString) => {
        const enrollmentDate = dayjs.utc(enrollmentDateString);
        const today = dayjs.utc().startOf('day');
        const firstWorkDayIndex = workableDays.findIndex(d => d.isSame(enrollmentDate, 'day') || d.isAfter(enrollmentDate));
        if (firstWorkDayIndex === -1) return { actualStartDate: enrollmentDate, daysWorked: 0, firstWorkDayIndex: -1 };
        const actualStartDate = workableDays[firstWorkDayIndex];
        const studentPersonalCalendar = workableDays.slice(firstWorkDayIndex);
        const daysWorked = studentPersonalCalendar.filter(d => d.isSame(today, 'day') || d.isBefore(today)).length;
        return { actualStartDate, daysWorked, firstWorkDayIndex };
    };

    const calculateSkillsProgress = (daysWorked) => {
        const progress = competenciesData.map(skill => ({ ...skill, expectedHours: 0 }));
        const fullWeeks = Math.floor(daysWorked / 5);
        const partialDays = daysWorked % 5;
        for (let week = 0; week < fullWeeks; week++) {
            competenciesData.forEach((skill, i) => { progress[i].expectedHours += skill.weeklyHours[week] || 0; });
        }
        if (partialDays > 0 && fullWeeks < totalWeeks) {
            competenciesData.forEach((skill, i) => { progress[i].expectedHours += (skill.weeklyHours[fullWeeks] || 0) * (partialDays / 5); });
        }
        return progress;
    };

    const updateAllViews = (enrollmentDateString) => {
        if (!enrollmentDateString) { mainContent.hidden = true; summaryDiv.style.display = 'none'; return; }
        mainContent.hidden = false; summaryDiv.style.display = 'block';
        const { actualStartDate, daysWorked, firstWorkDayIndex } = calculateProgress(enrollmentDateString);
        
        if ((firstWorkDayIndex === -1 || actualStartDate.isAfter(dayjs.utc())) && !isSingleStudentMode) {
            const summaryText = `Le premier jour de travail de cet élève est prévu pour le <strong>${actualStartDate.format('D MMMM YYYY')}</strong>.`;
            summaryDiv.innerHTML = summaryText; lastStudentSummaryHTML = summaryText;
            skillsTable.innerHTML = ''; ganttChart.innerHTML = ''; scheduleView.innerHTML = '';
            return;
        }

        const totalHours = daysWorked * 6;
        const summaryText = `Depuis son premier jour le <strong>${actualStartDate.format('D MMMM YYYY')}</strong>, <strong>${daysWorked} j. ouvrables</strong> se sont écoulés.<br>Total d'heures attendues: <strong>${totalHours.toFixed(1)} heures</strong>.`;
        summaryDiv.innerHTML = summaryText; lastStudentSummaryHTML = summaryText;
        const skillsProgress = calculateSkillsProgress(daysWorked);
        updateProgressTab(skillsProgress, firstWorkDayIndex);
        updateTimelineTab(daysWorked);
        updateScheduleTab(firstWorkDayIndex);
    };

    const updateTabTitles = (name) => {
        const titles = document.querySelectorAll('h2[data-original-title]');
        titles.forEach(title => {
            const originalText = title.dataset.originalTitle;
            if (name) {
                if (title.id === 'schedule-title') {
                    // Laisser updateScheduleTab gérer ce titre
                } else {
                    title.innerHTML = `${originalText} de <strong>${name}</strong>`;
                }
            } else {
                title.innerHTML = originalText;
            }
        });
    };

    const updateProgressTab = (skillsProgress, studentFirstDayIndex) => {
        skillsTable.innerHTML = `<thead><tr><th>Compétence</th><th>Local</th><th>H. Totales</th><th>Statut</th><th>Détails</th></tr></thead><tbody></tbody>`;
        const tbody = skillsTable.querySelector('tbody');
        skillsProgress.forEach(skill => {
            const typeClass = skill.type.toLowerCase().startsWith('th') ? 'badge-theorique' : 'badge-pratique';
            let status = '', statusClass = '', details = '';
            const roundedHours = Math.round(skill.expectedHours);
            if (roundedHours <= 0) {
                status = 'À venir'; statusClass = 'not-started';
                const skillStartDayIndex = studentFirstDayIndex + (skill.startWeek * 5);
                if (workableDays[skillStartDayIndex]) { details = `Début prévu le: <strong>${workableDays[skillStartDayIndex].format('D MMMM YYYY')}</strong>`; }
                else { details = `Début prévu semaine: <strong>${skill.startWeek + 1}</strong>`; }
            } else if (roundedHours >= skill.totalHours) {
                status = 'Terminé'; statusClass = 'completed';
                details = `<strong>${roundedHours}h</strong> / ${skill.totalHours}h attendues.`;
            } else {
                status = 'En cours'; statusClass = 'in-progress';
                const skillEndDayIndex = studentFirstDayIndex + (skill.endWeek * 5) + 4;
                let endDateText = '';
                if (workableDays[skillEndDayIndex]) { endDateText = `<br>Fin prévue le: <strong>${workableDays[skillEndDayIndex].format('D MMMM YYYY')}</strong>`; }
                details = `<strong>${roundedHours}h</strong> / ${skill.totalHours}h attendues.${endDateText}`;
            }
            const row = tbody.insertRow();
            row.innerHTML = `<td><span class="competency-badge ${typeClass}">${skill.shortName}</span><span class="competency-name-text">${skill.name}</span></td><td>${skill.local}</td><td>${skill.totalHours}</td><td><span class="tag ${statusClass}">${status}</span></td><td>${details}</td>`;
        });
    };

    const updateTimelineTab = (daysWorked) => {
        ganttChart.innerHTML = '';
        const nameColWidth = 300;
        const weekColWidth = 35;
        ganttChart.style.minWidth = `${nameColWidth + (totalWeeks * weekColWidth)}px`;

        const namesCol = document.createElement('div'); namesCol.className = 'gantt-names-col';
        const timelinesCol = document.createElement('div'); timelinesCol.className = 'gantt-timelines-col';
        namesCol.innerHTML = `<div class="gantt-header-cell">Compétence</div>`;
        const headerTimeline = document.createElement('div'); headerTimeline.className = 'gantt-header-cell';
        const weeksHeader = document.createElement('div'); weeksHeader.className = 'gantt-weeks-header';
        weeksHeader.innerHTML = `<div class="gantt-week-header-title">Semaines</div>`;
        const weekNumbers = document.createElement('div'); weekNumbers.className = 'gantt-week-numbers';
        for (let i = 1; i <= totalWeeks; i++) { 
            weekNumbers.innerHTML += `<div class="gantt-week-number">${i}</div>`; 
        }
        weeksHeader.appendChild(weekNumbers); headerTimeline.appendChild(weeksHeader); timelinesCol.appendChild(headerTimeline);
        
        competenciesData.forEach(skill => {
            const typeClass = skill.type.toLowerCase().startsWith('th') ? 'badge-theorique' : 'badge-pratique';
            namesCol.innerHTML += `<div class="gantt-skill-name" title="${skill.name}"><span class="competency-badge ${typeClass}">${skill.shortName}</span><span class="gantt-skill-name-text">${skill.name}</span></div>`;
            const skillTimeline = document.createElement('div'); skillTimeline.className = 'gantt-timeline';
            const laneGrid = document.createElement('div'); laneGrid.className = 'gantt-lanes-grid';
            laneGrid.style.gridTemplateColumns = `repeat(${totalWeeks}, 1fr)`;

            const barTypeClass = skill.type.toLowerCase().startsWith('th') ? 'type-theorique' : 'type-pratique';
            laneGrid.innerHTML = `<div class="gantt-bar ${barTypeClass}" style="--start: ${skill.startWeek + 1}; --end: ${skill.endWeek + 2};">${skill.totalHours}h</div>`;
            skillTimeline.appendChild(laneGrid); timelinesCol.appendChild(skillTimeline);
        });

        const progressLine = document.createElement('div'); progressLine.id = 'gantt-progress-line';
        const progressInWeeks = daysWorked / 5;
        progressLine.style.left = `${(progressInWeeks / totalWeeks) * 100}%`;
        timelinesCol.appendChild(progressLine); 
        
        ganttChart.appendChild(namesCol); 
        ganttChart.appendChild(timelinesCol);
    };

    const updateScheduleTab = (studentFirstDayIndex) => {
        scheduleView.innerHTML = '';
        calendarHeader.innerHTML = '';
        const selectedMonth = monthSelector.value;

        if (!selectedMonth || studentFirstDayIndex === -1) {
            scheduleView.innerHTML = "<p>Sélectionnez un élève et un mois pour voir l'horaire.</p>";
            if (scheduleTitle) scheduleTitle.innerHTML = scheduleTitle.dataset.originalTitle;
            return;
        }

        const monthName = dayjs.utc(selectedMonth).format('MMMM YYYY');
        const capitalizedMonthName = monthName.charAt(0).toUpperCase() + monthName.slice(1);
        const originalTitle = scheduleTitle.dataset.originalTitle;
        const studentName = (studentSelect && studentSelect.value) ? studentSelect.options[studentSelect.selectedIndex].text : studentNameInput.value;
        scheduleTitle.innerHTML = `${originalTitle} de <strong>${studentName}</strong> pour <strong>${capitalizedMonthName}</strong>`;
        
        const today = dayjs.utc().startOf('day');
        const startOfMonth = dayjs.utc(selectedMonth).startOf('month');
        const endOfMonth = dayjs.utc(selectedMonth).endOf('month');
        const daysOfWeek = ['Lundi', 'Mardi', 'Mercredi', 'Jeudi', 'Vendredi', 'Samedi', 'Dimanche'];
        daysOfWeek.forEach(day => { calendarHeader.innerHTML += `<div class="calendar-header-day">${day}</div>`; });
        const firstDayOfMonth = startOfMonth.day();
        const paddingDays = (firstDayOfMonth === 0) ? 6 : firstDayOfMonth - 1;
        for (let i = 0; i < paddingDays; i++) { scheduleView.innerHTML += `<div class="day-card empty"></div>`; }
        for (let i = 1; i <= endOfMonth.date(); i++) {
            const currentDay = startOfMonth.date(i);
            const dayOfWeek = currentDay.day();
            if (dayOfWeek === 0 || dayOfWeek === 6) {
                scheduleView.innerHTML += `<div class="day-card weekend"><div class="day-header">${currentDay.date()}</div></div>`;
                continue;
            }
            const workableDay = workableDays.find(d => d.isSame(currentDay, 'day'));
            if (workableDay) {
                const absoluteDayIndex = workableDays.findIndex(wd => wd.isSame(workableDay, 'day'));
                const studentDayNumber = absoluteDayIndex - studentFirstDayIndex;
                if (studentDayNumber < 0) {
                    scheduleView.innerHTML += `<div class="day-card holiday"><div class="day-header">${currentDay.date()}</div><div class="day-body"></div></div>`;
                    continue;
                }
                const studentWeek = Math.floor(studentDayNumber / 5);
                const totalWeeklyHours = competenciesData.reduce((sum, s) => sum + (s.weeklyHours[studentWeek] || 0), 0);
                let dailyHTML = '';
                if (totalWeeklyHours > 0) {
                    competenciesData.forEach(skill => {
                        const weeklyHours = skill.weeklyHours[studentWeek] || 0;
                        if (weeklyHours > 0) {
                            const proportion = weeklyHours / totalWeeklyHours; const dailyHours = proportion * 6;
                            const typeClass = skill.type.toLowerCase().startsWith('th') ? 'badge-theorique' : 'badge-pratique';
                            dailyHTML += `<div class="competency-badge ${typeClass}"><span>${skill.shortName}</span><span class="competency-hours">${dailyHours.toFixed(1)}h</span></div>`;
                        }
                    });
                }
                const dayCard = document.createElement('div'); dayCard.className = 'day-card';
                if (currentDay.isSame(today, 'day')) dayCard.classList.add('today');
                dayCard.innerHTML = `<div class="day-header">${currentDay.date()}</div><div class="day-body">${dailyHTML || ''}</div>`;
                scheduleView.appendChild(dayCard);
            } else {
                scheduleView.innerHTML += `<div class="day-card holiday"><div class="day-header">${currentDay.date()}</div><div class="day-body">Congé</div></div>`;
            }
        }
    };

    const updateOccupationTab = () => {
        occupationCalendarGrid.innerHTML = '';
        const selectedLocal = localSelector.value;
        const selectedMonth = occupationMonthSelector.value;

        if (!selectedLocal || !selectedMonth) {
            occupationCalendarGrid.innerHTML = "<p style='grid-column: 1 / 8; text-align: center;'>Veuillez sélectionner un local et un mois pour afficher le calendrier d'occupation.</p>";
            return;
        }

        const today = dayjs.utc().startOf('day');
        const startOfMonth = dayjs.utc(selectedMonth).startOf('month');
        const endOfMonth = dayjs.utc(selectedMonth).endOf('month');
        
        let headerHTML = '';
        const daysOfWeek = ['Lundi', 'Mardi', 'Mercredi', 'Jeudi', 'Vendredi', 'Samedi', 'Dimanche'];
        daysOfWeek.forEach(day => { headerHTML += `<div class="calendar-header-day">${day}</div>`; });

        let bodyHTML = '';
        const firstDayOfMonth = startOfMonth.day();
        const paddingDays = (firstDayOfMonth === 0) ? 6 : firstDayOfMonth - 1;
        for (let i = 0; i < paddingDays; i++) { bodyHTML += `<div class="day-card empty"></div>`; }

        for (let i = 1; i <= endOfMonth.date(); i++) {
            const currentDay = startOfMonth.date(i);
            const dayOfWeek = currentDay.day();
            let cardClasses = 'day-card';
            let cardBody = '';

            if (dayOfWeek === 0 || dayOfWeek === 6) {
                cardClasses += ' weekend';
            } else {
                const workableDay = workableDays.find(d => d.isSame(currentDay, 'day'));
                if (workableDay) {
                    const studentsInRoom = new Set();
                    let totalStudentHoursInRoom = 0;

                    studentsData.forEach(student => {
                        const { firstWorkDayIndex } = calculateProgress(student.startDate);
                        if (firstWorkDayIndex === -1) return;

                        const absoluteDayIndex = workableDays.findIndex(d => d.isSame(currentDay, 'day'));
                        if (absoluteDayIndex < firstWorkDayIndex) return;

                        const studentDayNumber = absoluteDayIndex - firstWorkDayIndex;
                        const studentWeek = Math.floor(studentDayNumber / 5);
                        const totalWeeklyHoursForStudent = competenciesData.reduce((sum, s) => sum + (s.weeklyHours[studentWeek] || 0), 0);

                        if (totalWeeklyHoursForStudent > 0) {
                            competenciesData.forEach(skill => {
                                if (skill.local === selectedLocal) {
                                    const skillWeeklyHours = skill.weeklyHours[studentWeek] || 0;
                                    if (skillWeeklyHours > 0) {
                                        studentsInRoom.add(student.name);
                                        totalStudentHoursInRoom += (skillWeeklyHours / totalWeeklyHoursForStudent) * 6;
                                    }
                                }
                            });
                        }
                    });

                    if (studentsInRoom.size > 0) {
                        cardBody = `<div class="occupation-details">
                                        <div><strong>Nbr. élèves:</strong> ${studentsInRoom.size}</div>
                                        <div><strong>Élève*h =</strong> ${totalStudentHoursInRoom.toFixed(1)}</div>
                                    </div>`;
                    }
                } else {
                    cardClasses += ' holiday';
                    cardBody = `<div class="day-body">Congé</div>`;
                }
            }
             if (currentDay.isSame(today, 'day')) cardClasses += ' today';
            bodyHTML += `<div class="${cardClasses}"><div class="day-header">${currentDay.date()}</div>${cardBody}</div>`;
        }
        occupationCalendarGrid.innerHTML = headerHTML + bodyHTML;
    };
    
    const updateEvaluationsTab = () => {
        const evaluationsByMonday = {};
        studentsData.forEach(student => {
            const { daysWorked, firstWorkDayIndex } = calculateProgress(student.startDate);
            if (firstWorkDayIndex === -1) return;
            const studentSkillsProgress = calculateSkillsProgress(daysWorked);
            studentSkillsProgress.forEach((skillProgress, index) => {
                const originalCompetency = competenciesData[index];
                const roundedHours = Math.round(skillProgress.expectedHours);
                const isEnCours = roundedHours > 0 && roundedHours < originalCompetency.totalHours;
                if (isEnCours) {
                    const evalDayIndex = firstWorkDayIndex + (originalCompetency.endWeek * 5) + 4;
                    if (workableDays[evalDayIndex]) {
                        const evalDate = workableDays[evalDayIndex];
                        const mondayOfWeek = evalDate.day(1);
                        const mondayKey = mondayOfWeek.format('YYYY-MM-DD');
                        if (!evaluationsByMonday[mondayKey]) { evaluationsByMonday[mondayKey] = { date: mondayOfWeek, competencies: {} }; }
                        if (!evaluationsByMonday[mondayKey].competencies[originalCompetency.name]) {
                            evaluationsByMonday[mondayKey].competencies[originalCompetency.name] = { shortName: originalCompetency.shortName, type: originalCompetency.type, students: [] };
                        }
                        evaluationsByMonday[mondayKey].competencies[originalCompetency.name].students.push(student.name);
                    }
                }
            });
        });
        evaluationsView.innerHTML = '';
        const sortedMondays = Object.keys(evaluationsByMonday).sort();
        if (sortedMondays.length === 0) { evaluationsView.innerHTML = '<p>Aucune évaluation à venir pour les compétences actuellement en cours.</p>'; return; }
        sortedMondays.forEach(mondayKey => {
            const weekData = evaluationsByMonday[mondayKey];
            const weekContainer = document.createElement('div'); weekContainer.className = 'evaluation-week';
            let tableRowsHTML = '';
            Object.keys(weekData.competencies).forEach(competencyName => {
                const compData = weekData.competencies[competencyName];
                const typeClass = compData.type.toLowerCase().startsWith('th') ? 'badge-theorique' : 'badge-pratique';
                tableRowsHTML += `<tr><td><span class="competency-badge ${typeClass}">${compData.shortName}</span><span class="competency-name-text">${competencyName}</span></td><td class="evaluation-students-list">${compData.students.join(', ')}</td></tr>`;
            });
            weekContainer.innerHTML = `<h3>Semaine du ${weekData.date.format('D MMMM YYYY')}</h3><table><thead><tr><th>Compétence</th><th>Élèves à évaluer</th></tr></thead><tbody>${tableRowsHTML}</tbody></table>`;
            evaluationsView.appendChild(weekContainer);
        });
    };

    const updateCohortTab = () => {
        if (!studentsData || !workableDays || studentsData.length === 0) {
            cohortTimelineView.innerHTML = "<p>Aucune donnée d'élève à afficher.</p>";
            return;
        }

        const sortedStudents = [...studentsData].sort((a, b) =>
            dayjs.utc(a.startDate).valueOf() - dayjs.utc(b.startDate).valueOf()
        );

        const timelineStart = workableDays[0];
        const timelineEnd = workableDays[workableDays.length - 1];
        const months = [];
        let currentMonth = timelineStart.startOf('month');

        while (currentMonth.isBefore(timelineEnd) || currentMonth.isSame(timelineEnd, 'month')) {
            months.push(currentMonth);
            currentMonth = currentMonth.add(1, 'month');
        }

        let headerHTML = '<th>Élève</th>';
        months.forEach(month => {
            const formattedMonth = month.format('MMMM YYYY');
            headerHTML += `<th>${formattedMonth.charAt(0).toUpperCase() + formattedMonth.slice(1)}</th>`;
        });

        let bodyHTML = '';
        sortedStudents.forEach(student => {
            const studentStartDate = dayjs.utc(student.startDate);
            bodyHTML += `<tr><td>${student.name}</td>`;

            months.forEach(month => {
                if (studentStartDate.isSame(month, 'month')) {
                    bodyHTML += `<td><span class="entry-date-marker">${studentStartDate.format('D')}</span></td>`;
                } else {
                    bodyHTML += '<td></td>';
                }
            });

            bodyHTML += '</tr>';
        });

        cohortTimelineView.innerHTML = `
            <table class="cohort-table">
                <thead>
                    <tr>${headerHTML}</tr>
                </thead>
                <tbody>
                    ${bodyHTML}
                </tbody>
            </table>
        `;
    };

    const updateStageTab = () => {
        const stageCompetency = competenciesData.find(c => c.type.toLowerCase() === 'stage');
        
        if (!stageCompetency) {
            stageView.innerHTML = "<p>Aucune compétence de type 'Stage' n'a été trouvée dans la matrice.</p>";
            return;
        }

        const studentsWithStageDates = studentsData.map(student => {
            const { firstWorkDayIndex } = calculateProgress(student.startDate);
            if (firstWorkDayIndex === -1) return null;

            const stageStartDayIndex = firstWorkDayIndex + (stageCompetency.startWeek * 5);
            const stageEndDayIndex = firstWorkDayIndex + (stageCompetency.endWeek * 5) + 4;

            const startDate = workableDays[stageStartDayIndex];
            const endDate = workableDays[stageEndDayIndex];

            if (!startDate || !endDate) return null;

            return { name: student.name, startDate, endDate };
        }).filter(Boolean);

        if (studentsWithStageDates.length === 0) {
            stageView.innerHTML = "<p>Aucun élève n'a de stage planifié pour le moment.</p>";
            return;
        }
        
        studentsWithStageDates.sort((a, b) => a.startDate.valueOf() - b.startDate.valueOf());
        const overallStartDate = studentsWithStageDates[0].startDate;
        const overallEndDate = studentsWithStageDates.reduce((max, s) => s.endDate.isAfter(max) ? s.endDate : max, studentsWithStageDates[0].endDate);
        const totalDurationDays = overallEndDate.diff(overallStartDate, 'day') + 1;

        let htmlContent = '';
        studentsWithStageDates.forEach(student => {
            const offsetDays = student.startDate.diff(overallStartDate, 'day');
            const durationDays = student.endDate.diff(student.startDate, 'day') + 1;

            const leftPercent = (offsetDays / totalDurationDays) * 100;
            const widthPercent = (durationDays / totalDurationDays) * 100;

            htmlContent += `
                <div class="stage-student-row">
                    <div class="stage-student-name">${student.name}</div>
                    <div class="stage-timeline-track">
                        <div class="stage-bar" style="left: ${leftPercent}%; width: ${widthPercent}%;">
                            ${student.startDate.format('D MMM YYYY')} → ${student.endDate.format('D MMM YYYY')}
                        </div>
                    </div>
                </div>
            `;
        });

        stageView.innerHTML = htmlContent;
    };

    generatePlanBtn.addEventListener('click', () => {
        const studentName = studentNameInput.value.trim();
        const studentDate = studentDateInput.value;

        if (!studentName || !studentDate) {
            alert("Veuillez entrer un nom et une date d'entrée.");
            return;
        }
        
        studentsData = [{ name: studentName, startDate: studentDate }];
        
        lastStudentSummaryHTML = `Plan de formation pour : <strong>${studentName}</strong>`;
        summaryDiv.innerHTML = lastStudentSummaryHTML;
        
        updateTabTitles(studentName);
        updateAllViews(studentDate);
    });

    tabButtons.forEach(button => {
        button.addEventListener("click", () => {
            tabButtons.forEach(btn => btn.classList.remove("active"));
            button.classList.add("active");
            document.querySelectorAll(".tab-pane").forEach(pane => pane.classList.toggle("active", pane.id === button.dataset.tab));
            
            const tabId = button.dataset.tab;

            if (isSingleStudentMode) {
                studentSelectionContainer.hidden = true;
            } else {
                studentSelectionContainer.hidden = (tabId === 'evaluations' || tabId === 'locaux' || tabId === 'cohorte' || tabId === 'stage');
            }

            if (!isSingleStudentMode && (tabId === 'cohorte' || tabId === 'stage' || tabId === 'evaluations' || tabId === 'locaux')) {
                summaryDiv.style.display = 'block';
                const messages = {
                    cohorte: "Voici la ligne du temps montrant la date d'entrée de chaque élève dans la formation.",
                    stage: "Cette vue montre la planification des stages en entreprise pour chaque élève.",
                    evaluations: "Ce tableau regroupe les évaluations à venir pour les compétences <strong>actuellement en cours</strong>.",
                    locaux: "Sélectionnez un local et un mois pour visualiser son taux d'occupation et sa charge horaire journalière."
                };
                summaryDiv.innerHTML = messages[tabId];
                if (tabId === 'locaux') updateOccupationTab();
            } else {
                if ((studentSelect && studentSelect.value) || studentNameInput.value) {
                    summaryDiv.style.display = 'block';
                    summaryDiv.innerHTML = lastStudentSummaryHTML;
                } else {
                    summaryDiv.style.display = 'none';
                }
            }
        });
    });

    studentSelect.addEventListener('change', (e) => {
        const selectedOption = e.target.options[e.target.selectedIndex];
        const studentName = selectedOption.text;
        
        if (e.target.value) {
            updateTabTitles(studentName);
            updateAllViews(e.target.value);
        } else {
            updateTabTitles('');
            if(scheduleTitle) scheduleTitle.innerHTML = scheduleTitle.dataset.originalTitle;
            mainContent.hidden = true;
        }
    });
    
    monthSelector.addEventListener('change', () => {
        const studentDate = (studentSelect && studentSelect.value) || studentDateInput.value;
        if (studentDate) {
            const { firstWorkDayIndex } = calculateProgress(studentDate);
            updateScheduleTab(firstWorkDayIndex);
        }
    });

    if (localSelector) localSelector.addEventListener('change', updateOccupationTab);
    if (occupationMonthSelector) occupationMonthSelector.addEventListener('change', updateOccupationTab);

    scrollLeftBtn.addEventListener('click', () => { ganttContainer.scrollBy({ left: -300, behavior: 'smooth' }); });
    scrollRightBtn.addEventListener('click', () => { ganttContainer.scrollBy({ left: 300, behavior: 'smooth' }); });
    
    document.querySelectorAll('.print-btn').forEach(button => {
        button.addEventListener('click', () => {
            window.print();
        });
    });

    const addDragToScroll = (element) => {
        if (!element) return;
        let isDown = false;
        let startX;
        let scrollLeft;

        element.addEventListener('mousedown', (e) => {
            isDown = true;
            element.classList.add('active');
            startX = e.pageX - element.offsetLeft;
            scrollLeft = element.scrollLeft;
        });

        element.addEventListener('mouseleave', () => {
            isDown = false;
            element.classList.remove('active');
        });

        element.addEventListener('mouseup', () => {
            isDown = false;
            element.classList.remove('active');
        });

        element.addEventListener('mousemove', (e) => {
            if (!isDown) return;
            e.preventDefault();
            const x = e.pageX - element.offsetLeft;
            const walk = (x - startX);
            element.scrollLeft = scrollLeft - walk;
        });
    };

    addDragToScroll(ganttContainer);
    addDragToScroll(cohortTimelineView);

    startApplication();
});